package ru.yandex.search.messenger.proxy.chatmedia;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;

import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.messenger.proxy.config.ImmutableMoxyConfig;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;

class ChatMediaRequestContext {
    private static final FieldsTermsSupplierFactory MESSAGE_LINKS_FIELDS =
        new FieldsTermsSupplierFactory("message_links", "message_first_link_snippet", "message_first_link_title");
    private static final FieldsTermsSupplierFactory MESSAGE_FILE_FIELDS =
        new FieldsTermsSupplierFactory("message_filename_p","message_filename_tokenized_p");

    private final ChatMediaHandler chatMediaHandler;
    private final Collection<String> get;
    private final boolean localityShuffle;
    private final boolean allowLaggingHosts;
    private final long failoverDelay;
    private final User user;
    private final String chatId;
    private final String messagesService;
    private final int prev;
    private final int next;
    private final long near;
    private final JsonType jsonType;
    private final boolean base64;
    private final Collection<String> excludeModerationActions;
    private final Collection<String> types;
    private final Collection<String> flags;
    private final Long clearTs;
    private final boolean printData;
    private final SearchRequestText request;
    private final FieldsTermsSupplierFactory fieldsFactory;
    private final boolean multipleLinks;

    ChatMediaRequestContext(
        final ChatMediaHandler chatMediaHandler,
        final ProxySession session,
        final ImmutableMoxyConfig config,
        final Long clearTs)
        throws HttpException
    {
        this.clearTs = clearTs;
        this.messagesService = config.messagesService();
        this.chatMediaHandler = chatMediaHandler;
        this.get = session.params().getAll(
            ChatMediaHandler.GET,
            Collections.emptySet(),
            ChatMediaHandler.SET_PARSER);
        this.printData = session.params().getBoolean("print-data", true);
        this.localityShuffle = session.params().getBoolean(
            "locality-shuffle",
            config.topPostsLocalityShuffle());
        this.failoverDelay = session.params().getLong(
            "failover-delay",
            config.topPostsFailoverDelay());
        this.allowLaggingHosts = session.params().getBoolean(
            "allow-lagging-hosts",
            ChatMediaHandler.ALLOW_LAGGING_HOSTS);
        String requestString = session.params().getString("request", "");
        this.request =
            SearchRequestText.parseSuggest(requestString, false);
        types = session.params().getAll(
            "types",
            ChatMediaHandler.DEFAULT_MESSAGE_TYPES,
            ChatMediaHandler.SET_PARSER);
        if (types.contains("link")) {
            this.fieldsFactory = MESSAGE_LINKS_FIELDS;
        } else if (types.contains("file_message")) {
            this.fieldsFactory = MESSAGE_FILE_FIELDS;
        } else {
            this.fieldsFactory = null;
        }
        Collection<String> flags = Collections.emptySet();
        if (types.contains("important_message")) {
            types.remove("important_message");
            if (flags.isEmpty()) {
                flags = new LinkedHashSet<>();
            }
            flags.add(MessageFields.IMPORTANT.prefixed());
        }

        if (types.contains("link")) {
            types.remove("link");
            if (flags.isEmpty()) {
                flags = new LinkedHashSet<>();
            }
            flags.add(MessageFields.HAS_LINKS.prefixed());
        }

        this.flags = flags;

        this.excludeModerationActions = session.params().getAll(
            "exclude-moderation-actions",
            ChatMediaHandler.DEFAULT_EXCLUDE_MODERATION,
            ChatMediaHandler.SET_PARSER);
        this.base64 = session.params().getBoolean("base64", true);
        this.chatId = session.params().getString("chat-id");
        this.prev = session.params().getInt("prev", ChatMediaHandler.DEFAULT_LENGTH);
        this.next = session.params().getInt("next", ChatMediaHandler.DEFAULT_LENGTH);
        this.near = session.params().getLong("near");
        this.user = new User(messagesService, new StringPrefix(chatId));
        this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        this.multipleLinks = session.params().getBoolean("multiple_links", false);
    }

    public Long clearTs() {
        return clearTs;
    }

    public boolean localityShuffle() {
        return localityShuffle;
    }

    public boolean allowLaggingHosts() {
        return allowLaggingHosts;
    }

    public long failoverDelay() {
        return failoverDelay;
    }

    public User user() {
        return user;
    }

    public String chatId() {
        return chatId;
    }

    public int prev() {
        return prev;
    }

    public int next() {
        return next;
    }

    public long near() {
        return near;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public boolean base64() {
        return base64;
    }

    public Collection<String> excludeModerationActions() {
        return excludeModerationActions;
    }

    public Collection<String> types() {
        return types;
    }

    public Collection<String> flags() {
        return flags;
    }

    public Collection<String> get() {
        return get;
    }

    public boolean printData() {
        return printData;
    }

    public SearchRequestText request() {
        return request;
    }

    public FieldsTermsSupplierFactory fieldsFactory() {
        return fieldsFactory;
    }

    public boolean multipleLinks() {
        return multipleLinks;
    }
}
