package ru.yandex.search.messenger.proxy.forward;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.MapParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.ps.search.messenger.ContactFields;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserId;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.ScopeContext;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.StringPrefix;

public class ForwardSuggestHandler implements ProxyRequestHandler {
    public static final boolean ALLOW_LAGGING_HOSTS = true;
    private static final CollectionParser<String, Set<String>, Exception> SETS_PARSER =
            new CollectionParser<>(String::trim, HashSet::new, '\n');
    private static final GroupAndPvpChats EMPTY_ENTRY =
        new GroupAndPvpChats(Collections.emptyList(), Collections.emptyList(), null);

    private static final MapParser<String, Long> MAP_PARSER =
        new MapParser<>(
            LinkedHashMap::new,
            NonEmptyValidator.TRIMMED,
            PositiveLongValidator.INSTANCE);

    private static final String USER_FIELDS =
        UserFields.DATA.stored() + ',' + UserFields.ID.stored();

    private static final String EXTRACT_USER_DATA_FIELDS =
        USER_FIELDS
            + ',' + UserFields.BLACKLISTED_USERS.stored()
            + ',' + UserFields.HIDDEN_PVP_CHATS.stored()
            + ',' + UserFields.SEARCH_PRIVACY.stored();

    private static final String GROUP_FIELDS =
            ChatFields.ID.stored() + ','
            + ChatFields.LAST_MESSAGE_TIMESTAMP.stored() + ','
            + ChatFields.MESSAGE_COUNT.stored() + ','
            + ChatFields.DATA.stored();
    private static final String PVP_FIELDS = GROUP_FIELDS + ',' + USER_FIELDS;
    private static final String ZEN = "zen";

    private final Moxy moxy;
    private final ForwardsCache forwardsCache;

    public ForwardSuggestHandler(final Moxy moxy) {
        this.moxy = moxy;
        this.forwardsCache = new ForwardsCache(moxy);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        ForwardRequestContext context = new ForwardRequestContext(session, moxy);
        ForwardPrinter callback = new ForwardPrinter(context);

        List<UserOrChatInfo> cached = forwardsCache.get(context);
        if (cached != null) {
            context.session().logger().info("From cache " + cached.size());
            callback.completed(cached);
            return;
        }

        context.callback(new CachingCallback(callback, context));

        DoubleFutureCallback<Map.Entry<JsonObject, JsonObject>, JsonObject>
            chatsAndContactsCallback = new DoubleFutureCallback<>(
                new ChatsAndContactsCallback(context));

        DoubleFutureCallback<JsonObject, JsonObject> chatsCallback =
                new DoubleFutureCallback<>(chatsAndContactsCallback.first());

        context.sendRequest(new PvpFetchContext(context), chatsCallback.first());
        context.sendRequest(new GroupFetchContext(context), chatsCallback.second());
        context.sendRequest(new ContactsContext(context), chatsAndContactsCallback.second());
    }

    private class CachingCallback extends FilterFutureCallback<List<UserOrChatInfo>> {
        private final ForwardRequestContext context;

        public CachingCallback(
            final FutureCallback<? super List<UserOrChatInfo>> callback,
            final ForwardRequestContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<UserOrChatInfo> result) {
            super.completed(result);
            forwardsCache.put(context, result);
        }
    }

    private static class ChatsAndContactsCallback
        extends AbstractProxySessionCallback<
            Map.Entry<Map.Entry<JsonObject, JsonObject>, JsonObject>>
    {
        private final ForwardRequestContext context;

        public ChatsAndContactsCallback(
            final ForwardRequestContext context) {
            super(context.session());
            this.context = context;
        }

        private Map<String, String> parseContacts(final JsonObject contactsObj)
            throws JsonException
        {
            JsonList hits = contactsObj.asMap().getList("hitsArray");
            if (hits.size() == 0) {
                return Collections.emptyMap();
            }

            Map<String, String> map = new LinkedHashMap<>(hits.size());
            for (JsonObject hit : hits) {
                JsonMap item = hit.asMap();
                String contactId = item.getString(ContactFields.ID.stored());
                String contactName = item.getString(ContactFields.NAME.stored(), null);
                if (contactId != null && contactName != null) {
                    map.put(contactId, contactName);
                }
            }

            return map;
        }

        protected GroupAndPvpChats parseChats(
            final JsonObject pvps,
            final JsonObject groups)
            throws JsonException
        {
            JsonList pvpsList = pvps.asMap().getList("hitsArray");
            JsonList groupsList = groups.asMap().getList("hitsArray");
            if (pvpsList.size() <= 0 && groupsList.size() <= 0) {
                return EMPTY_ENTRY;
            }

            List<UserOrChatInfo> pvpResult = new ArrayList<>(pvpsList.size());
            List<UserOrChatInfo> groupResult = new ArrayList<>(groupsList.size());
            SimpleJsonParser parser = ForwardRequestContext.getJsonParser();
            String selfChatId = context.guid() + '_' + context.guid();
            UserOrChatInfo selfChat = null;

            try {
                for (JsonObject hitObj: pvpsList) {
                    JsonMap hit = hitObj.asMap();
                    context.convertUserOrChatInfo(hit, parser);
                    String chatId = hit.getString(ChatFields.ID.stored());
                    if (selfChatId.equals(chatId)) {
                        selfChat = new UserOrChatInfo(chatId, hit, ForwardEntityType.SELF_CHAT);
                        continue;
                    }
                    String userId = hit.getString(UserFields.ID.stored(), null);
                    if (userId != null) {
                        pvpResult.add(UserOrChatInfo.pvpChat(userId, hit));
                    }
                }
                for (JsonObject hitObj: groupsList) {
                    JsonMap hit = hitObj.asMap();
                    context.convertUserOrChatInfo(hit, parser);
                    String chatId = hit.getString(ChatFields.ID.stored());
                    if (chatId.startsWith("1/")) {
                        continue;
                    }
                    groupResult.add(UserOrChatInfo.chatInfo(chatId, hit));
                }
            } finally {
                ForwardRequestContext.freeJsonParser(parser);
            }
            return new GroupAndPvpChats(groupResult, pvpResult, selfChat);
        }

        @Override
        public void completed(
            final Map.Entry<Map.Entry<JsonObject, JsonObject>, JsonObject> result)
        {
            Map<String, String> contacts;
            GroupAndPvpChats chatsAndPvpChats;
            JsonObject pvpResult = result.getKey().getKey();
            JsonObject groupResult = result.getKey().getValue();
            JsonObject contactResult = result.getValue();
            if (context.debug()) {
                context.session().logger().info(
                    "Pvp Chats Response "
                        + JsonType.HUMAN_READABLE.toString(pvpResult));
                context.session().logger().info(
                    "Group Chats Response "
                        + JsonType.HUMAN_READABLE.toString(groupResult));
                context.session().logger().info(
                    "Contacts Response "
                        + JsonType.HUMAN_READABLE.toString(contactResult));
            }
            try {
                chatsAndPvpChats = parseChats(pvpResult, groupResult);
                contacts = parseContacts(contactResult);
            } catch (JsonException je) {
                failed(je);
                return;
            }

            if (context.debug()) {
                context.session().logger().info(
                    "GroupChats " + chatsAndPvpChats.groupChats());
                context.session().logger().info(
                    "Pvps " + chatsAndPvpChats.pvpChats());
                context.session().logger().info(
                    "Contacts " + contacts);
            }

            if (chatsAndPvpChats.groupChats().size() <= 0
                && chatsAndPvpChats.pvpChats().size() <= 0
                && contacts.size() <= 0)
            {
                context.callback().completed(Collections.emptyList());
                return;
            }

            UsersOnlineCallback uoCallback =
                    new UsersOnlineCallback(context, contacts, chatsAndPvpChats);
            if (contacts.size() > 0) {
                try {
                    // now we fetching blacklist and hiddenchats and online status
                    context.moxy().lastSeenInfoProvider().get(
                            session,
                            contacts.keySet(),
                            uoCallback);
                } catch (HttpException | IOException e) {
                    failed(e);
                }
            } else {
                uoCallback.completed(Collections.emptyMap());
            }
        }
    }

    private static class UsersOnlineCallback
            extends AbstractProxySessionCallback<Map<String, Long>>
    {
        private final ForwardRequestContext context;
        private final Map<String, String> contacts;
        private final GroupAndPvpChats chats;

        public UsersOnlineCallback(
                final ForwardRequestContext context,
                final Map<String, String> contacts,
                final GroupAndPvpChats chats)
        {
            super(context.session());

            this.context = context;
            this.contacts = contacts;
            this.chats = chats;
        }

        @Override
        public void completed(final Map<String, Long> onlineMap) {
            for (UserOrChatInfo pvp: chats.pvpChats()) {
                Long ts = onlineMap.get(pvp.id());
                if (ts != null) {
                    pvp.lastSeen(ts);
                }
            }
            List<Map.Entry<String, Long>> fromOnlines = Collections.emptyList();
            int left = (2 * context.length()) - chats.pvpChats().size();

            if (!context.sort().equals(ZEN) && left > 0 && onlineMap.size() > 0) {
                List<Map.Entry<String, Long>> onlinesList
                    = new ArrayList<>(onlineMap.entrySet());
                onlinesList.sort(OnlineUsersComparator.INSTANCE);
                fromOnlines = onlinesList.subList(0, Math.min(onlinesList.size(), left));
            }

            if (context.debug()) {
                context.session().logger().info(
                    "Users from onlines " + fromOnlines);
            } else {
                context.session().logger().info(
                    "Users from onlines: " + fromOnlines.size()
                        + " From GroupChats " + chats.groupChats().size()
                        + " From PvpChats " + chats.pvpChats().size());
            }

            ExtractUserDataCallback callback =
                new ExtractUserDataCallback(context, contacts, fromOnlines, chats);
            try {
                context.sendRequest(
                    new ExtractUserDataContext(context, fromOnlines),
                    callback);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class ExtractUserDataCallback
            extends AbstractProxySessionCallback<JsonObject>
    {
        private final ForwardRequestContext context;
        private final Map<String, String> contacts;
        private final List<Map.Entry<String, Long>> fromOnline;
        private final GroupAndPvpChats chats;

        public ExtractUserDataCallback(
            final ForwardRequestContext context,
            final Map<String, String> contacts,
            final List<Map.Entry<String, Long>> fromOnline,
            final GroupAndPvpChats chats)
        {
            super(context.session());
            this.context = context;
            this.contacts = contacts;
            this.fromOnline = fromOnline;
            this.chats = chats;
        }

        private Map<String, JsonMap> parseUsers(
                final JsonObject resultObj)
                throws JsonException
        {
            JsonList hits = resultObj.asMap().getList("hitsArray");
            Map<String, JsonMap> result = new LinkedHashMap<>(hits.size());
            for (JsonObject hit: hits) {
                JsonMap map = hit.asMap();
                String userId = map.getString(UserFields.ID.stored());
                if (userId != null) {
                    result.put(userId, map);
                }
            }

            return result;
        }

        protected void process(
                final JsonObject resultObj,
                final SimpleJsonParser parser)
                throws JsonException
        {
            if (context.debug()) {
                context.session().logger().info(
                    "ExtractUserData " + JsonType.HUMAN_READABLE.toString(resultObj));
            }
            Set<String> blacklistedUsers = Collections.emptySet();
            Map<String, Long> hiddenChats = Collections.emptyMap();
            Map<String, JsonMap> userDatas = parseUsers(resultObj);
            JsonMap requestUserData = userDatas.get(context.guid());
            if (requestUserData != null) {
                try {
                    hiddenChats = requestUserData.get(
                        UserFields.HIDDEN_PVP_CHATS.stored(),
                        Collections.emptyMap(),
                        MAP_PARSER);
                } catch (Exception e) {
                    context.session().logger().log(
                        Level.WARNING,
                        "Failed to parse hidden chats",
                        e);
                }

                blacklistedUsers =
                    requestUserData.get(
                        UserFields.BLACKLISTED_USERS.stored(),
                        Collections.emptySet(),
                        SETS_PARSER);
            }

            if (context.debug()) {
                context.session().logger().info(
                    "Blacklist: " + blacklistedUsers
                        + " Hidden: " + hiddenChats);
            }

            filterPvpChats(hiddenChats);
            filterContacts(blacklistedUsers, userDatas, parser);
            Set<String> contactPrivacyUsers = getContactPrivacyUsers(userDatas);

            ContactPrivacyFilterCallback cpfCallback
                = new ContactPrivacyFilterCallback(
                    context,
                    contacts,
                    chats,
                    userDatas,
                    fromOnline,
                    contactPrivacyUsers);

            if (contactPrivacyUsers.size() > 0) {
                if (context.debug()) {
                    context.session().logger().info(
                        "Checking by contacts privacy " + contactPrivacyUsers);
                } else {
                    context.session().logger().info(
                        "Checking by contacts privacy " + contactPrivacyUsers.size());
                }
                context.moxy().privacyFilter().filter(
                        context.session(),
                        context.guid(),
                        contactPrivacyUsers,
                        cpfCallback);
            } else {
                cpfCallback.completed(Collections.emptySet());
            }
        }

        private void filterPvpChats(final Map<String, Long> hiddenChats)
            throws JsonException
        {
            int filtered = 0;
            Iterator<UserOrChatInfo> pvpIterator = chats.pvpChats().iterator();
            while (pvpIterator.hasNext()) {
                UserOrChatInfo pvpInfo = pvpIterator.next();
                pvpInfo.data().remove(ChatFields.ID.stored());
                Long lastMesTs = pvpInfo.data().getLong(ChatFields.LAST_MESSAGE_TIMESTAMP.stored(), 0L);
                if (hiddenChats.getOrDefault(pvpInfo.id(), -1L) >= lastMesTs) {
                    filtered += 1;
                    if (context.debug()) {
                        context.session().logger().info(
                            "Removing pvp by hidden " + pvpInfo.id());
                    }
                    pvpIterator.remove();
                }
            }
            if (filtered != 0) {
                context.session().logger().info("Filtered by hidden " + filtered);
            }
        }

        private void filterContacts(
            final Set<String> blacklistedUsers,
            final Map<String, JsonMap> userDatas,
            final SimpleJsonParser parser)
            throws JsonException
        {
            int filteredByBlacklist = 0;
            int filteredByNoUserData = 0;
            int filteredDisplayRestricted = 0;
            Iterator<Map.Entry<String, Long>> iterator = fromOnline.iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, Long> entry = iterator.next();
                String guid = entry.getKey();
                if (blacklistedUsers.contains(guid)) {
                    filteredByBlacklist += 1;
                    iterator.remove();
                    if (context.debug()) {
                        context.session().logger().info(
                            "Filter by blacklist " + guid);
                    }
                    continue;
                }
                JsonMap userData = userDatas.get(guid);
                if (userData == null) {
                    filteredByNoUserData += 1;
                    iterator.remove();
                    if (context.debug()) {
                        context.session().logger().info(
                            "Filter by no user data " + guid);
                    }
                    continue;
                }
                context.convertUserOrChatInfo(userData, parser);
                JsonMap map = userData.getMap("user_data");
                if (map == null
                    || map.getBoolean("is_display_restricted", false))
                {
                    filteredDisplayRestricted += 1;
                    iterator.remove();
                    if (context.debug()) {
                        context.session().logger().info(
                            "Filter by display restricted " + guid);
                    }
                }
            }
            if (filteredByBlacklist > 0) {
                context.session().logger().info(
                    "Filtered by blacklist " + filteredByBlacklist);
            }
            if (filteredByNoUserData > 0) {
                context.session().logger().info(
                    "Filtered by no user data " + filteredByNoUserData);
            }
            if (filteredDisplayRestricted > 0) {
                context.session().logger().info(
                    "Filtered by display restricted " + filteredDisplayRestricted);
            }
        }

        private Set<String> getContactPrivacyUsers(
            final Map<String, JsonMap> userDatas)
            throws JsonException
        {
            Set<String> result = new HashSet<>(fromOnline.size());
            for (Map.Entry<String, Long> entry: fromOnline) {
                String onlineUser = entry.getKey();
                JsonMap userData = userDatas.get(onlineUser);
                String privacy = userData.remove(
                    UserFields.SEARCH_PRIVACY.stored()).asStringOrNull();
                if (SearchPrivacy.parse(privacy) == SearchPrivacy.CONTACTS) {
                    result.add(onlineUser);
                }
            }
            return result;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            SimpleJsonParser jsonParser = ForwardRequestContext.getJsonParser();
            try {
                process(resultObj, jsonParser);
            } catch (JsonException e) {
                failed(e);
            } finally {
                ForwardRequestContext.freeJsonParser(jsonParser);
            }
        }
    }

    private static class ContactPrivacyFilterCallback
            extends AbstractFilterFutureCallback<Set<String>, List<UserOrChatInfo>>
    {
        private final Map<String, String> contacts;
        private final GroupAndPvpChats chats;
        private final Map<String, JsonMap> userDatas;
        private final List<Map.Entry<String, Long>> fromOnline;
        private final Set<String> forClearance;
        private final ForwardRequestContext context;

        public ContactPrivacyFilterCallback(
            final ForwardRequestContext context,
            final Map<String, String> contacts,
            final GroupAndPvpChats chats,
            final Map<String, JsonMap> userDatas,
            final List<Map.Entry<String, Long>> fromOnline,
            final Set<String> forClearance)
        {
            super(context.callback());
            this.contacts = contacts;
            this.chats = chats;
            this.userDatas = userDatas;
            this.fromOnline = fromOnline;
            this.forClearance = forClearance;
            this.context = context;
        }

        @Override
        public void completed(final Set<String> success) {
            filterContacts(success);
            try {
                List<UserOrChatInfo> result;
                if (context.sort().equals(ZEN)) {
                    result = generateZenResult();
                } else {
                    result = generateResult();
                }
                addContactNames(result);
                callback.completed(result);
            } catch (JsonException e) {
                failed(e);
            }
        }

        private void filterContacts(final Set<String> success) {
            if (forClearance.size() > 0) {
                if (context.debug()) {
                    context.session().logger().info(
                        "Privacy contacts clearance, from " + forClearance
                            + " success: " + success);
                } else {
                    context.session().logger().info(
                        "Privacy contacts clearance, from " + forClearance.size()
                            + " success: " + success.size());
                }
            }
            fromOnline.removeIf(entry ->
                forClearance.contains(entry.getKey())
                    && !success.contains(entry.getKey()));
        }

        private List<UserOrChatInfo> generateResult() throws JsonException {
            List<UserOrChatInfo> result = new ArrayList<>(context.length());
            Set<String> guids = new HashSet<>(context.length());
            for (UserOrChatInfo pvpChat: chats.pvpChats()) {
                if (result.size() >= context.length()) {
                    return result;
                }
                result.add(pvpChat);
                guids.add(pvpChat.id());
            }
            result.addAll(chats.groupChats().subList(0, Math.min(
                context.length() - result.size(),
                chats.groupChats().size())));
            if (result.size() >= context.length()) {
                return result;
            }
            if (chats.selfChat() != null) {
                result.add(chats.selfChat());
                guids.add(context.guid());
            }
            for (Map.Entry<String, Long> entry: fromOnline) {
                if (result.size() >= context.length()) {
                    break;
                }
                String guid = entry.getKey();
                if (guids.contains(guid)) {
                    if (context.debug()) {
                        context.session().logger().info(
                            "Filter contact by pvp chat existing " + guid);
                    }
                    continue;
                }
                JsonMap userData = userDatas.get(guid);
                userData.remove(UserFields.HIDDEN_PVP_CHATS.stored());
                userData.remove(UserFields.SEARCH_PRIVACY.stored());
                userData.remove(UserFields.BLACKLISTED_USERS.stored());
                UserOrChatInfo info =
                    UserOrChatInfo.userInfo(guid, userData, entry.getValue());
                result.add(info);
            }
            return result;
        }

        private List<UserOrChatInfo> generateZenResult() throws JsonException {
            List<UserOrChatInfo> result = new ArrayList<>(context.length());
            Set<String> guids = new HashSet<>(context.length());
            int filteredByOld = 0;
            for (UserOrChatInfo pvpChat: chats.pvpChats()) {
                if (result.size() >= context.length()) {
                    return result;
                }
                if (isChatOld(pvpChat)) {
                    filteredByOld += 1;
                    if (context.debug()) {
                        context.session().logger().info(
                            "Filter by old chat " + pvpChat.id());
                    }
                } else {
                    result.add(pvpChat);
                    guids.add(pvpChat.id());
                }
            }
            if (filteredByOld > 0) {
                context.session().logger().info(
                    "Filtered by old pvp chat " + filteredByOld);
            }
            result.addAll(chats.groupChats().subList(0, Math.min(
                context.length() - result.size(),
                chats.groupChats().size())));
            if (result.size() >= context.length()) {
                return result;
            }
            if (chats.selfChat() != null) {
                result.add(chats.selfChat());
                guids.add(context.guid());
            }
            for (UserOrChatInfo pvpChat: chats.pvpChats()) {
                if (result.size() >= context.length()) {
                    break;
                }
                if (!guids.contains(pvpChat.id())
                    && contacts.containsKey(pvpChat.id()))
                {
                    result.add(UserOrChatInfo.userInfo(
                        pvpChat.id(),
                        pvpChat.data(),
                        pvpChat.lastSeen()));
                }
            }
            return result;
        }

        private void addContactNames(final List<UserOrChatInfo> list) {
            for (UserOrChatInfo info: list) {
                if (info.dataType().equals(ForwardEntityType.PVP_CHAT)
                    || info.dataType().equals(ForwardEntityType.USER))
                {
                    String name = contacts.get(info.id());
                    if (name != null) {
                        info.data().put("contact_name", new JsonString(name));
                    }
                }
            }
        }

        private boolean isChatOld(final UserOrChatInfo chat) throws JsonException {
            Long lastMesTs = chat.data().getLong(
                ChatFields.LAST_MESSAGE_TIMESTAMP.stored(),
                0L);
            return lastMesTs < lastMonthMicroseconds();
        }
    }

    private enum OnlineUsersComparator implements Comparator<Map.Entry<String, Long>> {
        INSTANCE;

        @Override
        public int compare(final Map.Entry<String, Long> o1, final Map.Entry<String, Long> o2) {
            return -Long.compare(o1.getValue(), o2.getValue());
        }
    }

    private static class ExtractUserDataContext extends ScopeContext<ForwardRequestContext> {
        private final User user;
        private final List<Map.Entry<String, Long>> users;

        public ExtractUserDataContext(
            final ForwardRequestContext context,
            final List<Map.Entry<String, Long>> users)
            throws HttpException
        {
            super(context.session(), context);

            this.users = users;
            user = new User(context.usersService(), new LongPrefix(0L));
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public String service() {
            return user.service();
        }

        @Override
        public QueryConstructor createQuery(
                final ForwardRequestContext context)
                throws HttpException
        {
            StringBuilder sb = new StringBuilder();
            sb.append("id:");
            UserId.appendId(sb, context.guid(), "0");
            if (users.size() > 0) {
                sb.append(" OR (");
                sb.append("id:(");
                for (Map.Entry<String, Long> user: users) {
                    UserId.appendId(sb, user.getKey(), "0");
                    sb.append(' ');
                }
                sb.setLength(sb.length() - 1);
                sb.append(')');
                sb.append(" AND NOT ");
                sb.append(UserFields.SEARCH_PRIVACY.global());
                sb.append(':');
                sb.append(SearchPrivacy.NOBODY.value());
                sb.append(" AND NOT ");
                sb.append(UserFields.IS_ROBOT.global());
                sb.append(":true");
                sb.append(")");
            }

            QueryConstructor query =
                    new QueryConstructor(
                            "/search?forward&IO_PRIO=0&json-type=dollar");
            query.append("get", EXTRACT_USER_DATA_FIELDS);
            query.append("prefix", "0");
            query.append("service", context.usersService());
            query.append("text",  sb.toString());

            if (context.debug()) {
                context.session().logger().info(this.getClass().getName() + " " + query.toString());
            }
            return query;
        }
    }

    private static class PvpFetchContext
        extends ScopeContext<ForwardRequestContext>
    {
        private final User user;

        public PvpFetchContext(final ForwardRequestContext context)
            throws HttpException
        {
            super(context.session(), context);
            user = new User(context.chatsService(), new LongPrefix(0L));
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public String service() {
            return user.service();
        }

        @Override
        public QueryConstructor createQuery(
            final ForwardRequestContext context)
            throws HttpException
        {
            QueryConstructor query =
                    new QueryConstructor(
                            "/search?forward&IO_PRIO=0&json-type=dollar");
            query.append("get", PVP_FIELDS);
            query.append("collector", "sorted");
            query.append("prefix", "0");
            query.append("service", context.chatsService());
            String text = "chat_members:" + context.guid()
                + " AND NOT chat_show_on_morda:true AND NOT chat_channel:true";
            query.append("text", text);
            query.append("postfilter", "user_is_robot != true");
            if (!context.sort().equals(ZEN)) {
                query.append("postfilter",
                    ChatFields.LAST_MESSAGE_TIMESTAMP.stored() + " >= "
                        + lastMonthMicroseconds());
            }
            query.append("dp", "contains(chat_id,/ slash)");
            query.append("dp", "filter_cmp(slash,!=,1)");
            query.append(
                    "dp",
                    "replace_all(chat_id,(" + context.guid() + "|_), guid_tmp)");
            query.append("dp", "equals(guid_tmp,,  empty)");
            query.append("dp", "const(non_existing_guid guid_ne)");
            query.append("dp", "if(empty,guid_ne,guid_tmp guid)");
            query.append(
                    "dp",
                    "left_join(guid,user_id,,user_data user_data,user_id"
                        + " user_id,user_is_robot user_is_robot)");
            query.append("sort", ChatFields.LAST_MESSAGE_TIMESTAMP.stored());
            query.append("length", context.length());
            if (context.debug()) {
                context.session().logger().info(
                    " Pvp query: " + query.toString());
            }
            return query;
        }
    }

    private static class GroupFetchContext
        extends ScopeContext<ForwardRequestContext>
    {
        private final User user;

        public GroupFetchContext(final ForwardRequestContext context)
            throws HttpException
        {
            super(context.session(), context);
            user = new User(context.chatsService(), new LongPrefix(0L));
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public String service() {
            return user.service();
        }

        @Override
        public QueryConstructor createQuery(
            final ForwardRequestContext context)
            throws HttpException
        {
            QueryConstructor query =
                new QueryConstructor(
                    "/search?forward&IO_PRIO=0&json-type=dollar");
            query.append("get", GROUP_FIELDS);
            query.append("collector", "sorted");
            query.append("prefix", "0");
            query.append("service", context.chatsService());
            String text = "chat_members:" + context.guid()
                + " AND NOT chat_show_on_morda:true AND NOT chat_channel:true";
            query.append("text", text);
            query.append("postfilter", "user_is_robot != true");
            query.append("postfilter",
                ChatFields.LAST_MESSAGE_TIMESTAMP.stored() + " >= "
                    + lastMonthMicroseconds());
            query.append("dp", "contains(chat_id,/ slash)");
            query.append("dp", "filter_cmp(slash,==,1)");
            query.append("sort", ChatFields.LAST_MESSAGE_TIMESTAMP.stored());
            query.append("length", context.length());
            if (context.debug()) {
                context.session().logger().info(
                    " Group query: " + query.toString());
            }
            return query;
        }
    }

    private static long lastMonthMicroseconds() {
        return (System.currentTimeMillis() - TimeUnit.DAYS.toMillis(31)) * 1000;
    }

    private static class ContactsContext extends ScopeContext<ForwardRequestContext> {
        private final User user;

        public ContactsContext(
                final ForwardRequestContext context)
                throws HttpException {
            super(context.session(), context);

            user = new User(context.messagesService(), new StringPrefix(context.guid()));
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public String service() {
            return user.service();
        }

        @Override
        public QueryConstructor createQuery(
                final ForwardRequestContext context)
                throws HttpException
        {
            QueryConstructor query =
                    new QueryConstructor(
                            "/search?forward&IO_PRIO=0&json-type=dollar");
            query.append("get", "contact_id,contact_name");
            query.append("prefix", context.guid());
            query.append("service", user.service());

            query.append("text", "contact_user_id:" + context.guid());
            query.append("length", context.maxContacts());

            if (context.debug()) {
                context.session().logger().info(
                        this.getClass().getName() + " " + query.toString());
            }
            return query;
        }
    }
}
