package ru.yandex.search.messenger.proxy.forward;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.logging.Level;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class ForwardsCache {
    private final Moxy moxy;
    private static final long MAX_SIZE = 20000;
    private static final long MAX_MILLIS_VALID = TimeUnit.MINUTES.toMillis(10);

    private final TimeFrameQueue<Long> cacheHit;
    private final Cache<CacheKey, Cached> cache;

    public ForwardsCache(final Moxy moxy) {
        this.moxy = moxy;

        cache = CacheBuilder.newBuilder()
            .maximumSize(MAX_SIZE)
            .concurrencyLevel(moxy.config().workers()).build();

        cache.stats();
        this.cacheHit = new TimeFrameQueue<>(moxy.config().metricsTimeFrame());
        moxy.registerStater(
            new PassiveStaterAdapter<>(
                cacheHit,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "forward-cache-hit_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "forward-cache-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
    }

    public List<UserOrChatInfo> get(final ForwardRequestContext context) {
        Cached cached = cache.getIfPresent(context.cacheKey());
        if (cached == null || cached.length < context.length()) {
            cacheHit.accept(0L);
            return null;
        }

        long diff = System.currentTimeMillis() - cached.ts();
        long queueId = getQueueId(context.guid(), MaxFunction.INSTANCE);
        if (queueId > cached.contactsQueueId) {
            return null;
        }

        if (cached.length > 0 && diff > MAX_MILLIS_VALID) {
            return null;
        }

        cacheHit.accept(1L);
        return Arrays.asList(cached.result);
    }

    protected long getQueueId(final String user, final Function<List<QueueHostInfo>, Long> func) {
        List<QueueHostInfo> infos;
        try {
            infos = moxy.producerClient().executeWithInfo(
                new User(moxy.config().messagesService(), new StringPrefix(user)))
                .get(100, TimeUnit.MILLISECONDS);
        } catch (ExecutionException | TimeoutException | InterruptedException e) {
            moxy.logger().log(Level.WARNING, "Failed to get forwards from cache", e);
            return Long.MIN_VALUE;
        }

        Long value = func.apply(infos);
        if (value == null) {
            value = Long.MIN_VALUE;
        }

        return value;
    }

    public void put(final ForwardRequestContext context, final List<UserOrChatInfo> infos) {
        long queueId = getQueueId(context.guid(), MinFunction.INSTANCE);
        if (queueId == Long.MIN_VALUE) {
            return;
        }

        UserOrChatInfo[] cacheValue = new UserOrChatInfo[infos.size()];
        cacheValue = infos.toArray(cacheValue);
        cache.put(context.cacheKey(), new Cached(cacheValue, queueId, context.length()));
    }

    private static final class Cached {
        private final UserOrChatInfo[] result;
        private final long contactsQueueId;
        private final long ts;
        private final int length;

        public Cached(
            final UserOrChatInfo[] result,
            final long contactsQueueId,
            final int length)
        {
            this.result = result;
            this.contactsQueueId = contactsQueueId;
            this.length = length;
            this.ts = System.currentTimeMillis();
        }

        public UserOrChatInfo[] result() {
            return result;
        }

        public long contactsQueueId() {
            return contactsQueueId;
        }

        public long ts() {
            return ts;
        }

        public int length() {
            return length;
        }
    }

    private static final class MaxFunction
        implements Function<List<QueueHostInfo>, Long>
    {
        public static final MaxFunction INSTANCE = new MaxFunction();

        @Override
        public Long apply(final List<QueueHostInfo> col) {
            if (col.isEmpty()) {
                return null;
            }

            long max = Long.MIN_VALUE;
            for (QueueHostInfo info: col) {
                if (info.queueId() > max) {
                    max = info.queueId();
                }
            }

            return max;
        }
    }

    private static final class MinFunction
        implements Function<List<QueueHostInfo>, Long>
    {
        public static final MinFunction INSTANCE = new MinFunction();

        @Override
        public Long apply(final List<QueueHostInfo> col) {
            if (col.isEmpty()) {
                return null;
            }

            long min = Long.MAX_VALUE;
            for (QueueHostInfo info: col) {
                if (info.queueId() < min && info.queueId() > 0) {
                    min = info.queueId();
                }
            }

            if (min == Long.MAX_VALUE) {
                min = -1;
            }

            return min;
        }
    }

    public static final class CacheKey {
        private final String guid;
        private final String sort;

        public CacheKey(final ForwardRequestContext context) {
            this.guid = context.guid();
            this.sort = context.sort();
        }

        @Override
        public boolean equals(Object obj) {
            return this == obj
                || (obj instanceof ForwardsCache.CacheKey
                        && Objects.equals(guid, ((ForwardsCache.CacheKey) obj).guid)
                        && Objects.equals(sort, ((ForwardsCache.CacheKey) obj).sort));
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(guid) ^ Objects.hashCode(sort);
        }
    }
}
