package ru.yandex.search.messenger.proxy.forward;

import java.io.IOException;

import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.ps.search.messenger.ChatFields;

public class UserOrChatInfo implements JsonValue {
    private final String id;
    private final JsonMap data;
    private final ForwardEntityType dataType;
    private long lastSeen = -1;

    public UserOrChatInfo(
        final String id,
        final JsonMap data,
        final ForwardEntityType dataType)
    {
        this.id = id;
        this.data = data;
        this.dataType = dataType;
    }

    public UserOrChatInfo(
        final String id,
        final JsonMap data,
        final ForwardEntityType dataType,
        final long lastSeen) {
        this.id = id;
        this.data = data;
        this.dataType = dataType;
        this.lastSeen = lastSeen;
    }

    public String id() {
        return id;
    }

    public JsonMap data() {
        return data;
    }

    public ForwardEntityType dataType() {
        return dataType;
    }

    public long lastSeen() {
        return lastSeen;
    }

    public UserOrChatInfo lastSeen(final long lastSeen) {
        this.lastSeen = lastSeen;
        return this;
    }

    public static UserOrChatInfo chatInfo(final String id, final JsonMap data) throws JsonException {
        return new UserOrChatInfo(
            id,
            data,
            ForwardEntityType.GROUP_CHAT,
            data.getLong(ChatFields.LAST_MESSAGE_TIMESTAMP.stored(), -1L));
    }

    public static UserOrChatInfo userInfo(final String id, final JsonMap data) {
        return new UserOrChatInfo(id, data, ForwardEntityType.USER);
    }

    public static UserOrChatInfo pvpChat(final String id, final JsonMap data) {
        return new UserOrChatInfo(id, data, ForwardEntityType.PVP_CHAT);
    }

    public static UserOrChatInfo userInfo(
        final String id,
        final JsonMap data,
        final long lastSeen) {
        return new UserOrChatInfo(id, data, ForwardEntityType.USER, lastSeen);
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        if (dataType == ForwardEntityType.GROUP_CHAT || dataType == ForwardEntityType.SELF_CHAT) {
            writer.key("chat_id");
            writer.value(id);
            if (dataType == ForwardEntityType.SELF_CHAT) {
                writer.key("saved_messages");
                writer.value(JsonBoolean.TRUE);
            }
            writer.key("chat_data");
            data.get("chat_data").writeValue(writer);
        } else if (dataType == ForwardEntityType.USER || dataType == ForwardEntityType.PVP_CHAT) {
            writer.key("user_id");
            writer.value(id);
            writer.key("user_data");
            data.get("user_data").writeValue(writer);
            String contactName = data.getString("contact_name", null);
            if (contactName != null) {
                writer.key("contact_name");
                writer.value(contactName);
            }
            if (lastSeen >= 0) {
                writer.key("last_online_ts");
                writer.value(lastSeen);
            }
        }
        writer.endObject();
    }

    @Override
    public String toString() {
        return "UserOrChatInfo{" +
            "id='" + id + '\'' +
            ", data=" + JsonType.NORMAL.toString(data) +
            ", dataType=" + dataType +
            ", lastSeen=" + lastSeen +
            '}';
    }
}
