package ru.yandex.search.messenger.proxy.recchannels;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpHost;
import org.apache.http.entity.StringEntity;

import ru.yandex.client.producer.ImmutableProducerClientConfig;
import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.ProducerClientConfigBuilder;
import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.messenger.proxy.config.ImmutableRecommendedChannelsConfig;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class RecommendedChannelsUpdater implements Runnable, Stater {
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(
            (s) -> s.toLowerCase(Locale.ENGLISH).trim(),
        LinkedHashSet::new,
        '\n');

    private final PrefixedLogger updateLogger;
    private final SharedConnectingIOReactor reactor;
    private final ProducerClient producerClient;
    private final AsyncClient backendClient;
    private final ImmutableRecommendedChannelsConfig config;
    private final Moxy moxy;
    private final int timeout;
    private final SimpleJsonParser jsonParser;
    private final String script;
    private volatile boolean stopped = false;
    private final AtomicReference<CachedResponse> recommendedChannels;
    private final CgiParams defaultParams;
    private final Thread thread;
    private final AtomicReference<DropStats> dropStats = new AtomicReference<>(new DropStats());

    public RecommendedChannelsUpdater(final Moxy moxy) throws IOException {
        moxy.registerStater(this);
        this.config = moxy.config().recommendedChannelsConfig();

        thread = new Thread(this, "RecChannelsThread");
        thread.setDaemon(true);
        reactor = new SharedConnectingIOReactor(moxy.config(), moxy.config().dnsConfig());
        reactor.start();
        ImmutableProducerClientConfig config;
        try {
            config = new ProducerClientConfigBuilder()
                .streaming(false)
                .connections(2)
                .host(moxy.config().producerClientConfig().host())
                .build();
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }

        defaultParams =
            new CgiParams("&length=" + 5 * RecommendedChannelsHandler.DEFAULT_LENGTH);

        this.moxy = moxy;
        this.timeout = (int) TimeUnit.SECONDS.toMillis(60);
        this.jsonParser = new SimpleJsonParser();
        this.updateLogger = moxy.logger().addPrefix("RecChannelsUpdater");

        this.script = IOStreamUtils.consume(
            new InputStreamReader(
                Moxy.class.getResourceAsStream("rec_channels_request.js"),
                StandardCharsets.UTF_8))
            .toString();

        backendClient = new AsyncClient(reactor, moxy.config().searchConfig());
        backendClient.start();
        producerClient =
            new ProducerClient(reactor, config, moxy.searchMap());
        producerClient.start();

        this.recommendedChannels =
            new AtomicReference<>(
                new CachedResponse(Collections.emptyList(), 0));

        try {
            process();
        } catch (Exception e) {
            throw new IOException("Failed to fetch recommended channels", e);
        }

        thread.start();
    }

    @Override
    public <E extends Exception> void stats(StatsConsumer<? extends E> statsConsumer) throws E {
        statsConsumer.stat(
            "rec-channels-result-count_axxx",
            recommendedChannels.get().chats().size());
        statsConsumer.stat(
            "rec-channels-result-count_annn",
            recommendedChannels.get().chats().size());

        DropStats stats = dropStats.get();
        statsConsumer.stat(
            "rec-channels-drop-last-mes-verdict_axxx",
            stats.droppedLastMessageVerdict);
        statsConsumer.stat(
            "rec-channels-drop-messages-verdict-ratio_axxx",
            stats.droppedLastMessagesVerdictRatio);
        statsConsumer.stat(
            "rec-channels-drop-no-chat-data_axxx",
            stats.droppedNoChatData);
        statsConsumer.stat(
            "rec-channels-drop-no-messages_axxx",
            stats.droppedNoMessages);
        statsConsumer.stat(
            "rec-channels-keep-after-last-mests-filter_axxx",
            stats.lastAfterLastMesTsFilter);
    }

    protected void process() throws Exception {
        RecommendedChannelsRequestContext topContext =
            new RecommendedChannelsRequestContext(
                new CgiParams(""),
                EmptyFutureCallback.INSTANCE,
                moxy.config());
        ChatsRequestContext chatsContext =
            new ChatsRequestContext(moxy, topContext);

        List<QueueHostInfo> info =
            producerClient.executeWithInfo(chatsContext.user()).get(timeout, TimeUnit.MILLISECONDS);

        List<HttpHost> hosts = info.stream().map((x) -> x.host()).collect(Collectors.toList());
        String query = chatsContext.query().toString();
        JsonObject chatsResponse = backendClient.execute(
            hosts,
            new BasicAsyncRequestProducerGenerator(query),
            System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE).get();
        processChatsResponse(topContext, chatsResponse.asMap().getList("hitsArray"));
    }

    protected Collection<String> filterByMessages(
        final Collection<String> channelsIds,
        final User user,
        final DropStats stats)
        throws Exception
    {
        if (channelsIds.isEmpty()) {
            return Collections.emptySet();
        }

        List<QueueHostInfo> info =
            producerClient.executeWithInfo(user).get(timeout, TimeUnit.MILLISECONDS);
        List<HttpHost> hosts = info.stream().map((x) -> x.host()).collect(Collectors.toList());

        String prefixes = StringUtils.join(channelsIds, "\",\"", "[\"", "\"]");
        String shardScript = script.replace("%PREFIXES%", prefixes);

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/execute?rec_channels_update",
                new StringEntity(shardScript, StandardCharsets.UTF_8));

        JsonObject messagesResponse = backendClient.execute(
            hosts,
            generator,
            System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE).get();

        JsonList hits = messagesResponse.asList();
        Set<String> result = new LinkedHashSet<>(hits.size());
        for (JsonObject itemObj : hits) {
            JsonMap item = itemObj.asMap();
            String chatId = item.getString("prefix");
            JsonList messagesList = item.getList("messages");
            if (messagesList.size() == 0) {
                stats.droppedNoMessages += 1;
                continue;
            }

            JsonMap firstMessage = messagesList.get(0).asMap();
            Set<String> verdicts =
                firstMessage.get(
                    MessageFields.MODERATION_VERDICTS.stored(),
                    Collections.emptySet(),
                    SET_PARSER);

            if (!goodByVerdicts(verdicts)) {
                updateLogger.info(
                    "Dropping, last message, not good verdicts "
                        + chatId + " " + verdicts);
                stats.droppedLastMessageVerdict += 1;
                continue;
            }
//            String firstAction = messagesList.get(0).asMap().getString("message_moderation_action", null);
//            if (!"KEEP".equalsIgnoreCase(firstAction)) {
//                updateLogger.info(
//                    "Dropping, last message, not keep "
//                        + chatId + " " + firstAction);
//                continue;
//            }

            int keeps = 0;
            for (JsonObject postObj : messagesList) {
                verdicts =
                    postObj.asMap().get(
                        MessageFields.MODERATION_VERDICTS.stored(),
                        Collections.emptySet(),
                        SET_PARSER);
                if (goodByVerdicts(verdicts)) {
                    keeps += 1;
                }
//                String action = postObj.asMap().getString("message_moderation_action", null);
//                if ("KEEP".equalsIgnoreCase(action)) {
//                    keeps += 1;
//                }
            }
            float keepScore = keeps * 1.0f / messagesList.size();
            if (keepScore < 0.75f) {
                updateLogger.info("Dropping, low keep score " + chatId + " " + keepScore);
                stats.droppedLastMessagesVerdictRatio += 1;
                continue;
            }

            result.add(chatId);
        }

        return result;
    }

    protected boolean goodByVerdicts(final Set<String> verdicts) {
        boolean result = false;
        for (String verdict: config.goodVerdicts()) {
            if ("any".equalsIgnoreCase(verdict)) {
                return true;
            }

            result |= verdicts.contains(verdict);
        }

        return result;
    }

    protected void processChatsResponse(
        final RecommendedChannelsRequestContext context,
        final JsonList hits)
        throws Exception
    {
        DropStats dropStats = new DropStats();
        Map<String, JsonMap> chatsMap = new LinkedHashMap<>(hits.size());

        for (JsonObject o: hits) {
            JsonMap hit = o.asMap();
            String chatId = hit.getString(RecommendedChannelsHandler.CHAT_ID, null);
            if (chatId != null) {
                context.jsonReformat(hit, jsonParser);
                chatsMap.put(chatId, hit);
            }
        }

        dropStats.lastAfterLastMesTsFilter = chatsMap.size();

        Map<SearchMapShard, List<String>> shardMap = new HashMap<>();
        Map<SearchMapShard, User> userMap = new HashMap<>();
        for (String chatId: chatsMap.keySet()) {
            Prefix prefix = new StringPrefix(chatId);
            User user = new User(moxy.config().messagesService(), prefix);
            SearchMapShard shard = moxy.searchMap().apply(user);
            List<String> chats = shardMap.get(shard);
            if (chats == null) {
                chats = new ArrayList<>(2);
                shardMap.put(shard, chats);
                userMap.put(shard, user);
            }
            chats.add(chatId);
        }


        List<Chat> recChannels = new ArrayList<>();
        for (Map.Entry<SearchMapShard, List<String>> entry: shardMap.entrySet()) {
            Collection<String> channels =
                filterByMessages(entry.getValue(), userMap.get(entry.getKey()), dropStats);
            for (String channel: channels) {
                JsonMap chatData = chatsMap.get(channel);
                if (chatData != null) {
                    recChannels.add(new Chat(channel, chatData));
                } else {
                    updateLogger.warning("Skipping channel, no chat data " + channel);
                    dropStats.droppedNoChatData += 1;
                }
            }
        }

        this.dropStats.set(dropStats);
        updateLogger.info("Got channels, " + recChannels.size());
        updateLogger.info(recChannels.stream().map((x) -> x.chatId()).collect(Collectors.joining()));
        recommendedChannels.set(new CachedResponse(recChannels, System.currentTimeMillis()));
    }

    public CachedResponse channels() {
        return recommendedChannels.get();
    }

    @Override
    public void run() {
        try {
            while (!stopped) {
                try {
                    process();
                } catch (Exception e) {
                    updateLogger.log(
                        Level.WARNING,
                        "Failed to update channels",
                        e);
                }

                Thread.sleep(TimeUnit.MINUTES.toMillis(10));
            }
        } catch (InterruptedException ie) {
            updateLogger.log(
                Level.WARNING,
                "Update thread interrupted",
                ie);
        }
    }

    public void close() throws IOException {
        reactor.close();
        producerClient.close();
        backendClient.close();
    }

    private static final class DropStats {
        protected int droppedLastMessageVerdict = 0;
        protected int droppedLastMessagesVerdictRatio = 0;
        protected int droppedNoChatData = 0;
        protected int lastAfterLastMesTsFilter = 0;
        protected int droppedNoMessages = 0;
    }
}
