package ru.yandex.search.messenger.proxy.recents;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.function.GenericUnaryOperator;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.MapParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.ps.search.messenger.ContactFields;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.OnlineUsersComparator;

public class DefaultRecentsHandler implements ProxyRequestHandler {
    private static final CollectionParser<String, Set<String>, Exception> SETS_PARSER =
        new CollectionParser<>(String::trim, HashSet::new, '\n');
    private static final MapParser<String, Long> MAP_PARSER =
        new MapParser<>(
            LinkedHashMap::new,
            NonEmptyValidator.TRIMMED,
            PositiveLongValidator.INSTANCE);
    private static final JsonObject EMPTY_CONTACTS;
    static {
        JsonMap empty = new JsonMap(BasicContainerFactory.INSTANCE);
        empty.put("hitsArray", new JsonList(BasicContainerFactory.INSTANCE));
        EMPTY_CONTACTS = empty;
    }

    private final Moxy moxy;

    public DefaultRecentsHandler(final Moxy moxy) {
        this.moxy = moxy;
    }

    public void handle(final ProxySession session) throws HttpException {
        RecentRequestContext context = new RecentRequestContext(session, moxy);
        if (context.moxy().recentsCache() != null && context.allowCached()) {
            List<UserInfo> infos = context.moxy().recentsCache().get(context);
            if (infos != null) {
                context.session().logger().info("From cache");
                context.callback().completed(infos);
                return;
            }
        }
        DoubleFutureCallback<JsonObject, JsonObject> pvpAndContactsCallback =
            new DoubleFutureCallback<>(new PvpAndContactsCallback(context));
        context.sendRequest(new PvpsFetchContext(context), pvpAndContactsCallback.first());
        if (context.hasV2Org()) {
            pvpAndContactsCallback.second().completed(EMPTY_CONTACTS);
        } else {
            context.sendRequest(new ContactsContext(context), pvpAndContactsCallback.second());
        }
    }

    private static class PvpAndContactsCallback
        extends AbstractProxySessionCallback<Map.Entry<JsonObject, JsonObject>>
    {
        private final RecentRequestContext context;

        public PvpAndContactsCallback(
            final RecentRequestContext context)
        {
            super(context.session());
            this.context = context;
        }

        private Map<String, String> parseContacts(
            final JsonObject contactsObj)
            throws JsonException
        {
            JsonList hits = contactsObj.asMap().getList("hitsArray");
            if (hits.size() == 0) {
                return Collections.emptyMap();
            }

            Map<String, String> map = new LinkedHashMap<>(hits.size());
            for (JsonObject hit: hits) {
                JsonMap item = hit.asMap();
                String contactId = item.getString(ContactFields.ID.stored());
                String contactName = item.getString(ContactFields.NAME.stored(), null);
                if (contactId != null && contactName != null) {
                    map.put(contactId, contactName);
                }
            }
            map.remove(context.guid());
            return map;
        }

        protected Map<String, JsonMap> parsePvps(
            final JsonObject pvpsResult)
            throws JsonException
        {
            JsonList pvps = pvpsResult.asMap().getList("hitsArray");
            if (pvps.size() <= 0) {
                return Collections.emptyMap();
            }

            Map<String, JsonMap> result = new LinkedHashMap<>(pvps.size());
            for (JsonObject hitObj: pvps) {
                JsonMap hit = hitObj.asMap();
                String userId = hit.getString(UserFields.ID.stored(), null);
                if (userId != null) {
                    result.put(userId, hit);
                }
            }
            result.remove(context.guid());
            return result;
        }

        @Override
        public void completed(final Map.Entry<JsonObject, JsonObject> resultPair) {
            Map<String, String> contacts;
            Map<String, JsonMap> pvps;
            if (context.debug()) {
                context.session().logger().info(
                    "Pvps Response "
                        + JsonType.HUMAN_READABLE.toString(resultPair.getKey()));
                context.session().logger().info(
                    "Contacts Response "
                        + JsonType.HUMAN_READABLE.toString(resultPair.getValue()));
            }
            try {
                pvps = parsePvps(resultPair.getKey());
                contacts = parseContacts(resultPair.getValue());
            } catch (JsonException je) {
                failed(je);
                return;
            }

            if (context.debug()) {
                context.session().logger().info(
                    "Pvps " + pvps);
                context.session().logger().info(
                    "Contacts " + contacts);
            }

            if (pvps.size() <= 0 && contacts.size() <= 0) {
                context.callback().completed(Collections.emptyList());
                return;
            }

            UsersOnlineCallback uoCallback =
                new UsersOnlineCallback(context, contacts, pvps);
            if (contacts.size() > 0) {
                try {
                    // now we fetching blacklist and hiddenchats and online status
                    context.moxy().lastSeenInfoProvider().get(
                        session,
                        contacts.keySet(),
                        uoCallback);
                } catch (HttpException | IOException e) {
                    failed(e);
                }
            } else {
                uoCallback.completed(Collections.emptyMap());
            }
        }
    }

    private static class UsersOnlineCallback
        extends AbstractProxySessionCallback<Map<String, Long>>
    {
        private final RecentRequestContext context;
        private final Map<String, String> contacts;
        private final Map<String, JsonMap> pvps;

        public UsersOnlineCallback(
            final RecentRequestContext context,
            final Map<String, String> contacts,
            final Map<String, JsonMap> pvps)
        {
            super(context.session());

            this.context = context;
            this.contacts = contacts;
            this.pvps = pvps;
        }

        @Override
        public void completed(final Map<String, Long> onlineMap) {
            if (context.debug()) {
                context.session().logger().info(
                    "User Onlines " + onlineMap);
            }
            int left = 2 * context.length() - pvps.size();

            List<Map.Entry<String, Long>> onlinesList
                = new ArrayList<>(onlineMap.entrySet());
            onlinesList.sort(OnlineUsersComparator.INSTANCE);

            Set<String> fromOnlines = Collections.emptySet();
            if (left > 0) {
                fromOnlines = new LinkedHashSet<>(Math.min(onlinesList.size(), left));

                for (int i = 0; i < Math.min(onlinesList.size(), left); i++) {
                    Map.Entry<String, Long> item = onlinesList.get(i);
                    if (!pvps.containsKey(item.getKey())) {
                        fromOnlines.add(item.getKey());
                    }
                }
            }

            if (context.debug()) {
                context.session().logger().info(
                    "Took from onlines " + fromOnlines);
            } else {
                context.session().logger().info(
                    "From onlines: " + fromOnlines.size() + " From pvs " + pvps.size());
            }

            ExtractUserDataCallback callback =
                new ExtractUserDataCallback(context, contacts, fromOnlines, pvps);
            try {
                context.sendRequest(
                    new ExtractUserDataContext(context, fromOnlines),
                    callback);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class ExtractUserDataCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final RecentRequestContext context;
        private final Map<String, String> contacts;
        private final Set<String> fromOnline;
        private final Map<String, JsonMap> pvps;

        public ExtractUserDataCallback(
            final RecentRequestContext context,
            final Map<String, String> contacts,
            final Set<String> fromOnline,
            final Map<String, JsonMap> pvps)
        {
            super(context.session());
            this.context = context;
            this.contacts = contacts;
            this.fromOnline = fromOnline;
            this.pvps = pvps;
        }

        private Map<String, JsonMap> parseUsers(
            final JsonObject resultObj)
            throws JsonException
        {
            JsonList hits = resultObj.asMap().getList("hitsArray");
            Map<String, JsonMap> result = new LinkedHashMap<>(hits.size());
            for (JsonObject hit: hits) {
                JsonMap map = hit.asMap();
                String userId = map.getString(UserFields.ID.stored());
                if (userId != null) {
                    result.put(userId, map);
                }
            }

            return result;
        }

        protected void process(
            final JsonObject resultObj,
            final SimpleJsonParser parser)
            throws JsonException
        {
            if (context.debug()) {
                context.session().logger().info(
                    "ExtractUserData " + JsonType.HUMAN_READABLE.toString(resultObj));
            }
            Set<String> blacklistedUsers = Collections.emptySet();
            Map<String, Long> hiddenChats = Collections.emptyMap();
            Map<String, Long> clearedChats = Collections.emptyMap();
            Map<String, JsonMap> userDatas = parseUsers(resultObj);
            JsonMap requestUserData = userDatas.get(context.guid());
            if (requestUserData != null) {
                try {
                    hiddenChats = requestUserData.get(
                        UserFields.HIDDEN_PVP_CHATS.stored(),
                        Collections.emptyMap(),
                        MAP_PARSER);
                    String clearedHistoryChatsStr = requestUserData.getString(
                        UserFields.CLEARED_CHATS.stored(),
                        null);
                    if (clearedHistoryChatsStr != null) {
                        clearedChats = new MapParser<>(
                            LinkedHashMap::new,
                            new ClearedHistoryChatIdParser(context.guid()),
                            PositiveLongValidator.INSTANCE).apply(clearedHistoryChatsStr);
                    } else {
                        clearedChats = Collections.emptyMap();
                    }
                } catch (Exception e) {
                    context.session().logger().log(
                        Level.WARNING,
                        "Failed to parse hidden chats",
                        e);
                }

                blacklistedUsers =
                    requestUserData.get(
                        UserFields.BLACKLISTED_USERS.stored(),
                        Collections.emptySet(),
                        SETS_PARSER);
            }

            if (context.debug()) {
                context.session().logger().info(
                    "Blacklist: " + blacklistedUsers
                        + " Hidden: " + hiddenChats
                        + " ClearedHistoryChats: " + clearedChats);
            }
            int filteredByHidden = 0;
            int filteredByBlacklist = 0;
            int filteredByNoUserData = 0;
            int filteredDisplayRestricted = 0;

            List<UserInfo> result = new ArrayList<>(context.length());

            // add self user
            if (requestUserData != null) {
                context.convertUserInfo(requestUserData, parser);
                requestUserData.remove(UserFields.HIDDEN_PVP_CHATS.stored());
                requestUserData.remove(UserFields.CLEARED_CHATS.stored());
                requestUserData.remove(UserFields.SEARCH_PRIVACY.stored());
                requestUserData.remove(UserFields.BLACKLISTED_USERS.stored());
                UserInfo info = new UserInfo(context.guid(), requestUserData);
                result.add(info);
            }

            for (Map.Entry<String, JsonMap> entry: pvps.entrySet()) {
                entry.getValue().remove(ChatFields.ID.stored());
                String userId = entry.getValue().getOrNull(UserFields.ID.stored());

                if (userId == null) {
                    continue;
                }
                if (blacklistedUsers.contains(userId)) {
                    filteredByBlacklist += 1;
                    continue;
                }

                // we've chosen 0 and -1, because we want to NOT hide chat without messages

                Long lastMesTs = entry.getValue()
                    .getLong(ChatFields.LAST_MESSAGE_TIMESTAMP.stored(), 0L);
                if (hiddenChats.getOrDefault(userId, -1L) < lastMesTs
                    && clearedChats.getOrDefault(entry.getKey(), -1L) < lastMesTs)
                {
                    JsonMap userData = entry.getValue();
                    context.convertUserInfo(userData, parser);
                    JsonMap map = userData.getMapOrNull("user_data");
                    String contactName = contacts.get(entry.getKey());
                    if (contactName != null) {
                        userData.put("contact_name", new JsonString(contactName));
                    }

                    userData.remove(UserFields.HIDDEN_PVP_CHATS.stored());
                    userData.remove(UserFields.CLEARED_CHATS.stored());
                    userData.remove(UserFields.SEARCH_PRIVACY.stored());
                    userData.remove(UserFields.BLACKLISTED_USERS.stored());
                    result.add(new UserInfo(entry.getKey(), entry.getValue()));
                } else {
                    filteredByHidden += 1;
                }

                if (result.size() >= context.length()) {
                    context.callback().completed(result);
                    return;
                }
            }

            if (filteredByHidden != 0) {
                context.session().logger().info("Filtered by hidden " + filteredByHidden);
            }

            int left = context.length() - result.size();
            // ok, smth left, try online users;
            if (fromOnline.size() <= 0 || left <=0) {
                context.callback().completed(result);
                return;
            }

            int good = 0;

            Set<String> contactPrivacyUsers = new LinkedHashSet<>(fromOnline.size());
            Iterator<String> iterator = fromOnline.iterator();
            while (iterator.hasNext()) {
                String onlineUser = iterator.next();
                JsonMap userData = userDatas.get(onlineUser);
                boolean blacklisted = blacklistedUsers.contains(onlineUser);
                if (blacklisted) {
                    filteredByBlacklist += 1;
                    iterator.remove();
                    continue;
                }

                if (userData == null) {
                    iterator.remove();
                    filteredByNoUserData += 1;
                    continue;
                }

                context.convertUserInfo(userData, parser);
                JsonMap map = userData.getMap("user_data");
                if (map == null
                    || map.getBoolean("is_display_restricted", false))
                {
                    filteredDisplayRestricted += 1;
                    iterator.remove();
                    continue;
                }

                SearchPrivacy privacy =
                    SearchPrivacy.parse(
                        userData.remove(
                            UserFields.SEARCH_PRIVACY.stored())
                            .asStringOrNull());
                if (privacy == SearchPrivacy.CONTACTS) {
                    contactPrivacyUsers.add(onlineUser);
                } else {
                    good += 1;
                }

                if (contactPrivacyUsers.size() == 0 && left - good <= 0) {
                    break;
                }
            }

            if (filteredByBlacklist != 0) {
                context.session().logger().info("Filtered by blacklist " + filteredByBlacklist);
            }

            if (filteredByNoUserData != 0) {
                context.session().logger().info("Filtered by no user data " + filteredByNoUserData);
            }

            if (filteredDisplayRestricted > 0) {
                context.session().logger().info(
                    "Filtered by display restricted "
                        + filteredDisplayRestricted);
            }

            ContactPrivacyFilterCallback cpfCallback
                = new ContactPrivacyFilterCallback(
                context.callback(),
                result,
                contacts,
                userDatas,
                fromOnline,
                contactPrivacyUsers,
                context);

            if (contactPrivacyUsers.size() > 0) {
                context.session().logger().info(
                    "Checking by contacts privacy " + contactPrivacyUsers.size());
                context.moxy().privacyFilter().filter(
                    context.session(),
                    context.guid(),
                    contactPrivacyUsers,
                    cpfCallback);
            } else {
                cpfCallback.completed(Collections.emptySet());
            }
        }

        @Override
        public void completed(final JsonObject resultObj) {
            SimpleJsonParser jsonParser = RecentRequestContext.getJsonParser();
            try {
                process(resultObj, jsonParser);
            } catch (JsonException e) {
                failed(e);
            } finally {
                RecentRequestContext.freeJsonParser(jsonParser);
            }
        }
    }

    private static final class ClearedHistoryChatIdParser implements GenericUnaryOperator<String, Exception> {
        private final String requestGuid;

        public ClearedHistoryChatIdParser(final String requestGuid) {
            this.requestGuid = requestGuid;
        }

        @Override
        public String apply(final String s) throws Exception {
            String trimmed = NonEmptyValidator.TRIMMED.apply(s);
            int index = trimmed.indexOf('_');
            if (index > 0) {
                String first = trimmed.substring(0, index);
                if (first.equalsIgnoreCase(requestGuid)) {
                    return trimmed.substring(index + 1);
                } else {
                    return first;
                }
            }
            return trimmed;
        }
    }
}
