package ru.yandex.search.messenger.proxy.recents;

import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class RecentRequestContext {
    private static final int DEFAULT_MAX_USER_CONTACTS = 400;
    private static final int DEFAULT_ONLINE_LENGTH = 10;
    private static final int DEFAULT_LENGTH = 20;
    private static final boolean ALLOW_LAGGING_HOSTS = true;
    private static final String USER_DATA = "user_data";
    private static final String CHAT_DATA = "chat_data";

    private static final String GET = "get";
    private static final ConcurrentLinkedQueue<SimpleJsonParser> JSON_PARSERS =
            new ConcurrentLinkedQueue<>();

    private final int length;
    private final int maxContacts;
    private final int onlineLength;
    private final String get;
    private final boolean localityShuffle;
    private final boolean allowLaggingHosts;
    private final long failoverDelay;
    private final String guid;
    private final JsonType jsonType;
    private final ProxySession session;
    private final Moxy moxy;
    private final String userFilter;
    private final boolean debug;
    private final boolean allowCached;
    private final FutureCallback<List<UserInfo>> callback;
    private final String sort;
    private final Long v2Org;

    public RecentRequestContext(final ProxySession session, final Moxy moxy)
        throws HttpException
    {
        this.moxy = moxy;
        this.session = session;
        this.length = session.params().getInt("length", DEFAULT_LENGTH);
        this.v2Org = session.params().getLong("v2_org_id", null);
        String get =
            session.params().getString(GET, null);
        this.get = get;
        this.localityShuffle = session.params().getBoolean(
            "locality-shuffle",
            moxy.config().recentsLocalityShuffle());
        this.failoverDelay = session.params().getLong(
            "failover-delay",
            moxy.config().recentsFailoverDelay());
        this.maxContacts = session.params().getInt(
            "max-user-contacts",
            DEFAULT_MAX_USER_CONTACTS);
        this.onlineLength = session.params().getInt(
            "online-length",
            DEFAULT_ONLINE_LENGTH);

        this.allowCached = session.params().getBoolean("allow_cached", true);
        this.allowLaggingHosts = session.params().getBoolean(
            "allow-lagging-hosts",
            ALLOW_LAGGING_HOSTS);
        this.guid = session.params().getString("guid");
        this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        this.userFilter =
            session.params().getString(
                "user-filter",
                "user_org_id:0 AND user_is_dismissed:false AND user_is_robot:false");
        this.debug = session.params().getBoolean("debug", false);
        this.callback = new RecentsPrinter(this);
        this.sort = session.params().getString("sort", "");
    }

    public Long v2Org() {
        return v2Org;
    }

    public boolean hasV2Org() {
        return v2Org != null && v2Org != 0;
    }

    public FutureCallback<List<UserInfo>> callback() {
        return callback;
    }

    public boolean debug() {
        return debug;
    }

    public String userFilter() {
        return userFilter;
    }

    public Moxy moxy() {
        return moxy;
    }

    public int length() {
        return length;
    }

    public int maxContacts() {
        return maxContacts;
    }

    public String get() {
        return get;
    }

    public String guid() {
        return guid;
    }

    public boolean allowCached() {
        return allowCached;
    }

    public boolean localityShuffle() {
        return localityShuffle;
    }

    public boolean allowLaggingHosts() {
        return allowLaggingHosts;
    }

    public long failoverDelay() {
        return failoverDelay;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public ProxySession session() {
        return session;
    }

    public String usersService() {
        return moxy.config().usersService();
    }

    public String chatsService() {
        return moxy.config().chatsService();
    }

    public String messagesService() {
        return moxy.config().messagesService();
    }

    public int onlineLength() {
        return onlineLength;
    }

    public String sort() {
        return sort;
    }

    public void sendRequest(
            final ScopeContext context,
            final FutureCallback<JsonObject> callback)
            throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
                new PlainUniversalSearchProxyRequestContext(
                        context.user(),
                        null,
                        allowLaggingHosts(),
                        client,
                        session.logger());
        QueryConstructor query = context.query();
        query.append("service", context.service());
        moxy.sequentialRequest(
                session,
                requestContext,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                failoverDelay(),
                localityShuffle(),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                callback);
    }

    public static SimpleJsonParser getJsonParser() {
        SimpleJsonParser parser = JSON_PARSERS.poll();
        if (parser == null) {
            parser = new SimpleJsonParser();
        }
        return parser;
    }

    public void convertUserInfo(
            final JsonMap doc,
            final SimpleJsonParser jsonParser)
    {
        try {
            String jsonString = doc.get(CHAT_DATA).asStringOrNull();
            if (jsonString != null) {
                JsonObject obj = jsonParser.parse(jsonString);
                doc.put(CHAT_DATA, obj);
            }
            jsonString = doc.get(USER_DATA).asStringOrNull();
            if (jsonString != null) {
                JsonObject obj = jsonParser.parse(jsonString);
                doc.put(USER_DATA, obj);
            }
        } catch (JsonException e) { // skip, obj is null
        }
    }

    public static void freeJsonParser(final SimpleJsonParser parser) {
        JSON_PARSERS.add(parser);
    }
}
