package ru.yandex.search.messenger.proxy.recents;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.logging.Level;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class RecentsCache {
    private final Moxy moxy;
    private static final long MAX_MILLIS_VALID = TimeUnit.SECONDS.toMillis(200);
    private static final long MAX_SIZE = 150000;
    private final TimeFrameQueue<Long> cacheHit;
    private final Cache<String, Cached> cache;

    public RecentsCache(final Moxy moxy) {
        this.moxy = moxy;

        cache = CacheBuilder.newBuilder()
            .maximumSize(MAX_SIZE)
            .concurrencyLevel(moxy.config().workers()).build();

        cache.stats();
        this.cacheHit = new TimeFrameQueue<>(moxy.config().metricsTimeFrame());
        moxy.registerStater(
            new PassiveStaterAdapter<>(
                cacheHit,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "recents-cache-hit_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "recents-cache-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
    }

    public List<UserInfo> get(final RecentRequestContext context) {
        Cached cached = cache.getIfPresent(context.guid());
        if (cached == null || cached.length < context.length()) {
            cacheHit.accept(0L);
            return null;
        }

        long diff = System.currentTimeMillis() - cached.ts();
        long queueId = getQueueId(context.guid(), MaxFunction.INSTANCE);
        if (queueId > cached.contactsQueueId) {
            return null;
        }

        if (cached.length > 0 && diff > MAX_MILLIS_VALID) {
            return null;
        }

        cacheHit.accept(1L);
        return Arrays.asList(cached.result);
    }

    protected long getQueueId(final String user, final Function<List<QueueHostInfo>, Long> func) {
        List<QueueHostInfo> infos;
        try {
            infos = moxy.producerClient().executeWithInfo(
                new User(moxy.config().messagesService(), new StringPrefix(user)))
                .get(100, TimeUnit.MILLISECONDS);
        } catch (ExecutionException | TimeoutException | InterruptedException e) {
            moxy.logger().log(Level.WARNING, "Failed to get recents from cache", e);
            return Long.MIN_VALUE;
        }

        Long value = func.apply(infos);
        if (value == null) {
            value = Long.MIN_VALUE;
        }

        return value;
    }

    public void put(final RecentRequestContext context, final List<UserInfo> infos) {
        long queueId = getQueueId(context.guid(), MinFunction.INSTANCE);
        if (queueId == Long.MIN_VALUE) {
            return;
        }

        UserInfo[] cacheValue = new UserInfo[infos.size()];
        cacheValue = infos.toArray(cacheValue);
        cache.put(context.guid(), new Cached(cacheValue, queueId, context.length()));
    }

    private static final class Cached {
        private final UserInfo[] result;
        private final long contactsQueueId;
        private final long ts;
        private final int length;

        public Cached(
            final UserInfo[] result,
            final long contactsQueueId,
            final int length)
        {
            this.result = result;
            this.contactsQueueId = contactsQueueId;
            this.length = length;
            this.ts = System.currentTimeMillis();
        }

        public long ts() {
            return ts;
        }
    }

    private static final class MaxFunction
        implements Function<List<QueueHostInfo>, Long>
    {
        public static final MaxFunction INSTANCE = new MaxFunction();

        @Override
        public Long apply(final List<QueueHostInfo> col) {
            if (col.isEmpty()) {
                return null;
            }

            long max = Long.MIN_VALUE;
            for (QueueHostInfo info: col) {
                if (info.queueId() > max) {
                    max = info.queueId();
                }
            }

            return max;
        }
    }

    private static final class MinFunction
        implements Function<List<QueueHostInfo>, Long>
    {
        public static final MinFunction INSTANCE = new MinFunction();

        @Override
        public Long apply(final List<QueueHostInfo> col) {
            if (col.isEmpty()) {
                return null;
            }

            long min = Long.MAX_VALUE;
            for (QueueHostInfo info: col) {
                if (info.queueId() < min && info.queueId() > 0) {
                    min = info.queueId();
                }
            }

            if (min == Long.MAX_VALUE) {
                min = -1;
            }

            return min;
        }
    }
}
