package ru.yandex.search.messenger.proxy.suggest;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicSuggestRequestContext
    implements RequestProvider,
        SuggestRequestContext,
        SuggestRequestContextProvider
{
    private static final Long DEFAULT_SUGGEST_TIMEOUT = 900L;
    private static final CollectionParser<
        SuggestType,
        Set<SuggestType>,
        ? extends Exception>
        TYPES_PARSER = new CollectionParser<>(
            new EnumParser<>(SuggestType.class),
            () -> EnumSet.noneOf(SuggestType.class));

    private static final CollectionParser<
        String,
        Set<String>,
        ? extends Exception>
        NAMESPACES_PARSER =
        new CollectionParser<>(String::trim, LinkedHashSet::new);


    private final Moxy proxy;
    private final ProxySession session;
    private final Set<SuggestType> types;
    private final Set<String> namespaces;
    private final Set<String> namespacesGuids;
    private final Set<Set<SuggestType>> typeGroups;
    private final int length;
    private final boolean debug;
    private final Long timeout;
    private final JsonType jsonType;
    private final AsyncClient client;
    private final Supplier<? extends HttpClientContext> contextGenerator;
    private final String request;
    private final long lagTolerance;
    private final Long v2Org;
    private final ConcurrentMap<String, String> requestDeduplicator =
        new ConcurrentHashMap<String, String>();

    public BasicSuggestRequestContext(
        final Moxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        this.proxy = proxy;
        this.session = session;
        this.v2Org = session.params().getLong("v2_org_id", null);
        CgiParams params = session.params();
        List<String> suggestTypesList = params.getAll("suggest-types");
        Set<Set<SuggestType>> typeGroups = new LinkedHashSet<>();
        this.types = new LinkedHashSet<>();
        for (String st: suggestTypesList) {
            Set<SuggestType> subSet = new HashSet<>();
            try {
                subSet = TYPES_PARSER.apply(st);
                // kostyl
                boolean hasMesChats = subSet.remove(SuggestType.MESSAGES_CHATS);
                boolean hasMesPvp = subSet.remove(SuggestType.MESSAGES_PVP);
                if (hasMesChats || hasMesPvp) {
                    subSet.add(SuggestType.MESSAGES);
                    if (hasMesChats && !hasMesPvp) {
                        params.add("messages-group-chats", "true");
                        params.add("messages-pvp-chats", "false");
                    } else if (!hasMesChats && hasMesPvp) {
                        params.add("messages-group-chats", "false");
                        params.add("messages-pvp-chats", "true");
                    }

                    subSet.add(SuggestType.MESSAGES);
                }
                typeGroups.add(subSet);
                types.addAll(subSet);
            } catch (Exception e) {
                throw new BadRequestException(e);
            }
        }
        this.typeGroups = typeGroups;
//        types = params.get("suggest-types", TYPES_PARSER);
//        if (types.contains(SuggestType.USERS)
//            || types.contains(SuggestType.USERS_GLOBAL))
//        {
//            types.add(SuggestType.CONTACTS);
//        }
        length = params.get("length", PositiveIntegerValidator.INSTANCE);
        timeout = params.get(
            "timeout",
            DEFAULT_SUGGEST_TIMEOUT,
            DurationParser.POSITIVE_LONG);
        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        client = proxy.searchClient().adjust(session.context());
        contextGenerator =
            session.listener().createContextGeneratorFor(client);
        request = params.getString("request", "");
        String namespacesString = params.getString("namespaces",  null);
        if (namespacesString == null) {
            namespaces = null;
        } else {
            if (!namespacesString.isBlank()) {
                try {
                    namespaces = NAMESPACES_PARSER.apply(namespacesString);
                } catch (Exception e) {
                    throw new BadRequestException("Failed to parse namespace " + namespacesString, e);
                }
            } else {
                namespaces = Collections.emptySet();
            }
        }

        namespacesGuids = params.get("namespace_guids", Collections.emptySet(), NAMESPACES_PARSER);
        if (params.getBoolean("allow-lagging-hosts", true)) {
            lagTolerance = Long.MAX_VALUE;
        } else {
            lagTolerance = 0L;
        }
        debug = params.getBoolean("debug", false);
    }

    protected BasicSuggestRequestContext(
        final SuggestRequestContext other,
        final String request)
    {
        proxy = other.proxy();
        session = other.session();
        types = other.types();
        length = other.length();
        timeout = other.timeout();
        jsonType = other.jsonType();
        client = other.client();
        contextGenerator = other.contextGenerator();
        lagTolerance = other.lagTolerance();
        this.request = request;
        typeGroups = other.typeGroups();
        namespaces = other.namespaces();
        namespacesGuids = other.namespacesGuids();
        debug = other.debug();
        v2Org = other.v2Org();
    }

    @Override
    public Moxy proxy() {
        return proxy;
    }

    public Set<String> namespacesGuids() {
        return namespacesGuids;
    }

    @Override
    public ProxySession session() {
        return session;
    }

    @Override
    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    @Override
    public Set<Set<SuggestType>> typeGroups() {
        return typeGroups;
    }

    @Override
    public Set<SuggestType> types() {
        return types;
    }

    @Override
    public int length() {
        return length;
    }

    @Override
    public Long timeout() {
        return timeout;
    }

    @Override
    public JsonType jsonType() {
        return jsonType;
    }

    @Override
    public User user() {
        return null;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AsyncClient client() {
        return client;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    // RequestProvider implementation
    @Override
    public String request() {
        return request;
    }

    // SuggestRequestContextProvider implementation
    @Override
    public SuggestRequestContext suggestRequestContext() {
        return this;
    }

    @Override
    public long lagTolerance() {
        return lagTolerance;
    }

    @Override
    public boolean debug() {
        return debug;
    }

    @Override
    public boolean hadRequest(final SuggestType type, final String request) {
        String key = type.toString() + '#' + request;
        return requestDeduplicator.putIfAbsent(key, key) != null;
    }

    @Override
    public Set<String> namespaces() {
        return namespaces;
    }

    @Override
    public Long v2Org() {
        return v2Org;
    }
}

