package ru.yandex.search.messenger.proxy.suggest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.rules.DepartmentsSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.GroupsSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.LoginsSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.MergeResultsRule;
import ru.yandex.search.messenger.proxy.suggest.rules.MisspellRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveAllUsersRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveChatRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveContactsUsersRule;
import ru.yandex.search.messenger.proxy.suggest.rules.ResolveUserChatsRule;
import ru.yandex.search.messenger.proxy.suggest.rules.UsersPvp2Rule;
import ru.yandex.search.messenger.proxy.suggest.rules.UsersSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.chats.ChatsSuggestRule2;
import ru.yandex.search.messenger.proxy.suggest.rules.messages.MessagesSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestAllFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestInfosProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestUsersFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestsChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestsUsersFilterProvider;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;
import ru.yandex.search.rules.pure.TranslitRule2;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class SuggestHandler implements ProxyRequestHandler {
    private static final long TIMEOUT_OVERHEAD = 10L;
    private static final long MIN_TIMEOUT = 50L;
    private static final Supplier<List<SuggestItem>> NULL_SUPPLIER =
        () -> null;
    private static final String SUGGEST = " suggest";
    private static final TranslitRule.TableSelector TRANSLIT_SELECTOR =
        new TranslitSelector123();
    private static final TranslitRule2.TableSelector TRANSLIT_SELECTOR2 =
        new TranslitSelector123();

    private final Timer timer = new Timer("Suggest-Timer", true);
    private final Map<
        SuggestType,
        SearchRule<
            BasicSuggestRequestContext,
            List<SuggestItem>>> rules =
                new EnumMap<>(SuggestType.class);
    private final Map<SuggestType, TimeFrameQueue<Long>> failedSuggestStatMap;
    private final TimeFrameQueue<Long> failedTotal;

    private final Moxy proxy;

    //CSOFF: MethodLength
    public SuggestHandler(final Moxy proxy) {
        this.proxy = proxy;
        this.failedSuggestStatMap = new LinkedHashMap<>();
        // TODO: wrap every rule with TranslitRule and ConcatResultsRule
        //CSOFF: LineLength
        rules.put(
            SuggestType.USERS,
            new ResolveChatRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MisspellRule<>(
                                new ChainedSearchRule<>(
                                    new MergeResultsRule<>(
                                        new ChainedSearchRule<>(
                                            new UsersSuggestRule<>(proxy, SuggestType.USERS, false, false),
                                            (input, request) ->
                                                new SuggestRequestContextRequestAllFilterProvider(
                                                    input.suggestRequestContext(),
                                                    request,
                                                    AcceptAllUsersFilter.INSTANCE,
                                                    input.chatsFilter()))),
                                    (input, requests) ->
                                        new SuggestRequestContextRequestsChatsFilterProvider(
                                            input.suggestRequestContext(),
                                            input.chatsFilter(),
                                            requests))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsChatsFilterProvider(
                                    input.suggestRequestContext(),
                                    input.chatsFilter(),
                                    requests)),
                        true,
                        TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestChatsFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter))));
        rules.put(
            SuggestType.USERS_PVP,
            new TranslitRule<>(
                new ChainedSearchRule<>(
                    new MisspellRule<>(
                        new ChainedSearchRule<>(
                            new MergeResultsRule<>(
                                new ChainedSearchRule<>(
                                    new UsersPvp2Rule<>(proxy),
                                    (input, request) ->
                                        new SuggestRequestContextRequestAllFilterProvider(
                                            input.suggestRequestContext(),
                                            request,
                                            input.usersFilter(),
                                            AcceptAllChatsFilter.INSTANCE))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsUsersFilterProvider(
                                    input.suggestRequestContext(),
                                    null,
                                    requests))),
                    (input, requests) ->
                        new SuggestRequestContextRequestsUsersFilterProvider(
                            input.suggestRequestContext(),
                            null,
                            requests)),
                true,
                TRANSLIT_SELECTOR));
        rules.put(
            SuggestType.CONTACTS,
            new ResolveChatRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MisspellRule<>(
                                new ChainedSearchRule<>(
                                    new MergeResultsRule<>(
                                        new ChainedSearchRule<>(
                                            new ResolveContactsUsersRule<>(
                                                proxy,
                                                new ChainedSearchRule<>(
                                                    new UsersSuggestRule<>(proxy, SuggestType.CONTACTS, false, true),
                                                    (input, filter) ->
                                                        new SuggestRequestContextRequestAllFilterProvider(
                                                            input.suggestRequestContext(),
                                                            input.request(),
                                                            filter,
                                                            input.chatsFilter()))),
                                                (input, request) ->
                                                    new SuggestRequestContextRequestAllFilterProvider(
                                                        input.suggestRequestContext(),
                                                        request,
                                                        AcceptAllUsersFilter.INSTANCE,
                                                        input.chatsFilter()))),
                                    (input, requests) ->
                                        new SuggestRequestContextRequestsChatsFilterProvider(
                                            input.suggestRequestContext(),
                                            input.chatsFilter(),
                                            requests))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsChatsFilterProvider(
                                    input.suggestRequestContext(),
                                    input.chatsFilter(),
                                    requests)),
                            true,
                            TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestChatsFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter))));

        rules.put(
            SuggestType.USERS_GLOBAL,
            new ResolveAllUsersRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MisspellRule<>(
                                new ChainedSearchRule<>(
                                    new MergeResultsRule<>(
                                        new ChainedSearchRule<>(
                                            new UsersSuggestRule<>(proxy, SuggestType.USERS_GLOBAL, false, false),
                                            (input, request) ->
                                                new SuggestRequestContextRequestAllFilterProvider(
                                                    input.suggestRequestContext(),
                                                    request,
                                                    input.usersFilter(),
                                                    AcceptAllChatsFilter.INSTANCE))),
                                    (input, requests) ->
                                        new SuggestRequestContextRequestsUsersFilterProvider(
                                            input.suggestRequestContext(),
                                            input.usersFilter(),
                                            requests))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsUsersFilterProvider(
                                    input.suggestRequestContext(),
                                    input.usersFilter(),
                                    requests)),
                        true,
                        TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestUsersFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter))));
        rules.put(
            SuggestType.LOGIN,
            new LoginsSuggestRule<>(proxy));
        rules.put(
            SuggestType.GROUPS,
            new GroupsSuggestRule<>(proxy));
        rules.put(
            SuggestType.DEPARTMENTS,
            new DepartmentsSuggestRule<>(proxy));
        //CSON: LineLength

        //CSOFF: LineLength
        rules.put(
            SuggestType.MESSAGES,
            new ResolveUserChatsRule<>(
                new ChainedSearchRule<>(
                    new TranslitRule<>(
                        new ChainedSearchRule<>(
                            new MergeResultsRule<>(
                                new ChainedSearchRule<>(
                                    new MessagesSuggestRule<>(proxy),
                                    (input, request) ->
                                        new SuggestRequestContextRequestChatsFilterProvider(
                                            input.suggestRequestContext(),
                                            request,
                                            input.chatsFilter()))),
                            (input, requests) ->
                                new SuggestRequestContextRequestsChatsFilterProvider(
                                    input.suggestRequestContext(),
                                    input.chatsFilter(),
                                    requests)),
                        true,
                        TRANSLIT_SELECTOR),
                    (input, filter) ->
                        new SuggestRequestContextRequestChatsFilterProvider(
                            input.suggestRequestContext(),
                            input.request(),
                            filter))));
        //CSON: LineLength
        rules.put(
            SuggestType.CHATS,
            new TranslitRule2<>(
                new ChainedSearchRule<>(
                    new ChatsSuggestRule2<>(proxy, SuggestType.CHATS),
                    (input, requests) ->
                        new SuggestRequestContextRequestInfosProvider(
                            input.suggestRequestContext(),
                            requests)),
                true,
                TRANSLIT_SELECTOR2));

        rules.put(
            SuggestType.CHANNELS,
            new TranslitRule2<>(
                new ChainedSearchRule<>(
                    new ChatsSuggestRule2<>(proxy, SuggestType.CHANNELS),
                    (input, requests) ->
                        new SuggestRequestContextRequestInfosProvider(
                            input.suggestRequestContext(),
                            requests)),
                true,
                TRANSLIT_SELECTOR2));

        rules.put(
            SuggestType.CONTENT,
            (input, callback) -> callback.completed(Collections.emptyList()));
        failedTotal = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
                failedTotal,
                new NamedStatsAggregatorFactory<>(
                    "suggest-types-failed-total_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        for (SuggestType suggestType: rules.keySet()) {
            TimeFrameQueue<Long> failStat = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
            proxy.registerStater(
                new PassiveStaterAdapter<>(
                    failStat,
                    new NamedStatsAggregatorFactory<>(
                        "suggest-types-failed-"
                            + suggestType.name().toLowerCase(Locale.ENGLISH)
                            + "_ammm",
                        CountAggregatorFactory.INSTANCE)));
            failedSuggestStatMap.put(suggestType, failStat);
        }
    }
    //CSON: MethodLength

    @Override
    public void handle(final ProxySession session) throws HttpException {
        BasicSuggestRequestContext context =
            new BasicSuggestRequestContext(proxy, session);
        Map<SuggestType, List<SuggestItem>> responses =
            new EnumMap<>(SuggestType.class);
        FutureCallback<Object> printer =
            new IdempotentFutureCallback<>(
                new SuggestResultCallback(
                    // TODO: URGENT: html escaping for text
                    new SuggestPrinter(context, HtmlHighlighter.INSTANCE),
                    responses));
        MultiFutureCallback<Object> callback =
            new MultiFutureCallback<>(printer);
        List<FutureCallback<?>> typeCallbacks =
            new ArrayList<>(context.types().size());

        for (SuggestType type: context.types()) {
            ResponseStoringCallback typeCallback =
                new ResponseStoringCallback(
                    callback.newCallback(),
                    session.logger(),
                    type,
                    responses);

                typeCallbacks.add(typeCallback);
                rules.get(type).execute(context, typeCallback);
//            if (type == SuggestType.USERS
//                || type == SuggestType.USERS_GLOBAL)
//            {
//                rules.get(SuggestType.CONTACTS).execute(
//                    context,
//                    new ResponseStoringCallback(
//                        callback.newCallback(),
//                        session.logger(),
//                        SuggestType.CONTACTS,
//                        responses));
//            }
        }
        callback.done();
        Long timeoutBoxed = context.timeout();
        if (timeoutBoxed != null) {
            long now = System.currentTimeMillis();
            long timeElapsed = now - session.connection().requestStartTime();
            long timeout = timeoutBoxed.longValue() - timeElapsed;
            timeout = Math.max(timeout - TIMEOUT_OVERHEAD, MIN_TIMEOUT);
            timer.schedule(new TimeoutTask(typeCallbacks), timeout);
        }
    }

    private class ResponseStoringCallback
        extends ErrorSuppressingFutureCallback<List<SuggestItem>>
    {
        private final Logger logger;
        private final SuggestType type;
        private final Map<SuggestType, List<SuggestItem>> responses;
        private final TimeFrameQueue<Long> statByType;
        private volatile boolean done = false;

        // CSOFF: ParameterNumber
        ResponseStoringCallback(
            final FutureCallback<Object> callback,
            final Logger logger,
            final SuggestType type,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback, NULL_SUPPLIER);
            this.logger = logger;
            this.type = type;
            this.responses = responses;
            this.statByType = failedSuggestStatMap.get(type);
        }
        // CSON: ParameterNumber

        @Override
        public void failed(final Exception e) {
            logger.log(
                Level.WARNING,
                "Failed to get results for " + type + SUGGEST,
                e);

            if (!done) {
                statByType.accept(1L);
                failedTotal.accept(1L);
                super.failed(e);
            }
        }

        @Override
        public void completed(final List<SuggestItem> response) {
            if (response != null) {
                logger.info(
                    "Got " + response.size()
                    + " results for " + type + SUGGEST);
                synchronized (responses) {
                    List<SuggestItem> alreadyFound =
                        responses.get(type);
                    if (alreadyFound != null) {
                        alreadyFound.addAll(response);
                        Collections.sort(alreadyFound);
                    } else {
                        responses.put(type, response);
                    }
                }
            }

            //logger.log(Level.WARNING, "Completed " + response, new Exception("Exception"));
            done = true;
            callback.completed(null);
        }
    }

    private static class SuggestResultCallback
        extends AbstractFilterFutureCallback<Object, SuggestResult>
    {
        private final Map<SuggestType, List<SuggestItem>> responses;

        SuggestResultCallback(
            final FutureCallback<? super SuggestResult> callback,
            final Map<SuggestType, List<SuggestItem>> responses)
        {
            super(callback);
            this.responses = responses;
        }

        @Override
        public void completed(final Object response) {
            SuggestResult result;
            synchronized (responses) {
                result = new SuggestResult(responses);
            }

            callback.completed(result);
        }
    }

    private static class TimeoutTask extends TimerTask {
        private static final Exception TIMEOUT_EXC = new Exception("Timeouting by timer");
        private final List<FutureCallback<?>> callbacks;

        TimeoutTask(
            final List<FutureCallback<?>> callbacks)
        {
            this.callbacks = callbacks;
        }

        @Override
        public void run() {
            for (FutureCallback<?> callback: callbacks) {
                callback.failed(TIMEOUT_EXC);
            }
        }
    }

    private static final class TranslitSelector123
        implements TranslitRule.TableSelector, TranslitRule2.TableSelector
    {
        @Override
        public boolean useTable(
            final String request,
            final String table)
        {
            if (table.startsWith("translit") && request.length() < 2) {
//                System.err.println("Skipping translist for request: <"
//                    + request + '>');
                return false;
            }
            return true;
        }
    }
}

