package ru.yandex.search.messenger.proxy.suggest;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;

public class SuggestPrinter
    extends AbstractProxySessionCallback<SuggestResult>
{
    private static final String DEBUG = "debug";

    private final SuggestRequestContext context;
    private final boolean debug;
//    private final Highlighter highlighter;

    public SuggestPrinter(
        final SuggestRequestContext context,
        final Highlighter highlighter)
        throws BadRequestException
    {
        super(context.session());
        this.context = context;
        this.debug = context.session().params().getBoolean(DEBUG, false);
//        this.highlighter = highlighter;
    }

    private SuggestItem selectBest(final SuggestItem i1, final SuggestItem i2) {
        if (i1.score() >= i2.score()) {
            return i1;
        } else {
            return i2;
        }
    }

    @SuppressWarnings("StringSplitter")
    private void fillMatches(
        final List<String> matches,
        final String text,
        final String request)
    {
        String[] words = request.split("\\s+");
        String textLower = text.toLowerCase(Locale.ROOT);
        for (String word: words) {
            if (word.length() > 1 && word.charAt(0) == '@') {
                word = word.substring(1);
            }
            int index = textLower.indexOf(word.toLowerCase(Locale.ROOT));
            if (index != -1 && index < text.length()) {
//                System.err.println("MATCHED: word=" + word + ", text=" + text
//                    + ", index=" + index + ", len=" + word.length());
                matches.add(
                    text.substring(
                        index,
                        Math.min(index + word.length(), text.length())));
            }
        }
    }

    @Override
    public void completed(final SuggestResult result) {
        Set<SuggestType> failedSuggests = EnumSet.noneOf(SuggestType.class);
        Set<String> ids = new HashSet<>();
        Map<SuggestType, List<SuggestItem>> suggestResults = result.suggests();
        List<List<SuggestItem>> deduplicated =
            new ArrayList<>(Math.min(suggestResults.size(), context.length()));
        Map<Integer, List<List<SuggestItem>>> deduplicatedGrouped =
            new LinkedHashMap<>();
        int totalSize = 0;
        int group = 0;
        Map<Integer, List<SuggestItem>> groupedItems = new LinkedHashMap<>();
        for (Set<SuggestType> types: context.typeGroups()) {
            List<List<SuggestItem>> groupList = new ArrayList<>();
            deduplicatedGrouped.put(group, groupList);
            for (SuggestType type: types) {
                List<SuggestItem> suggestItems = suggestResults.get(type);
                if (suggestItems == null) {
                    failedSuggests.add(type);
                } else {
                    // Do not add new groups to deduplicated map if there is no
                    // space for this suggest group
                    // TODO: put all items to single list, replacing less
                    // scored with more scored items if they have same id,
                    // then sort items according to their score
                    if (deduplicated.size() < context.length()) {
                        LinkedHashMap<String, SuggestItem> innerDeduplicated =
                            new LinkedHashMap<>();
                        for (SuggestItem item: suggestItems) {
                            if (ids.contains(item.id())) {
                                continue;
                            }
                            SuggestItem other =
                                innerDeduplicated.get(item.id());
                            if (other != null) {
                                item = selectBest(item, other);
                            }
                            innerDeduplicated.put(item.id(), item);
                        }
                        List<SuggestItem> deduplicatedItems =
                            new ArrayList<>(innerDeduplicated.values());
                        ids.addAll(innerDeduplicated.keySet());
                        int size = deduplicatedItems.size();
                        if (size != 0) {
                            deduplicated.add(deduplicatedItems);
                            groupList.add(deduplicatedItems);
                            totalSize += size;
                        }
                    }
                }
            }
            group++;
        }
        // Leave at least 1 suggest item for each group
        if (context.typeGroups().size() > 1) {
            for (Map.Entry<Integer, List<List<SuggestItem>>> entry:
                deduplicatedGrouped.entrySet())
            {
                List<List<SuggestItem>> suggestGroup = entry.getValue();
                int groupSize = 0;
                for (List<SuggestItem> items: suggestGroup) {
                    groupSize += items.size();
                }
                int pos = suggestGroup.size() - 1;
                while (groupSize > context.length() && pos >= 0) {
                    List<SuggestItem> items = suggestGroup.get(pos);
                    int size = items.size();
                    if (size > 1) {
                        items.remove(size - 1);
                        --groupSize;
                    } else {
                        --pos;
                    }
                }
            }
        } else {
            int pos = deduplicated.size() - 1;
            while (totalSize > context.length()) {
                List<SuggestItem> items = deduplicated.get(pos);
                int size = items.size();
                if (size > 1) {
                    items.remove(size - 1);
                    --totalSize;
                } else {
                    --pos;
                }
            }
        }

        int printed = 0;
        StringBuilderWriter sbw = new StringBuilderWriter();
        StringBuilderWriter statSbw = new StringBuilderWriter();
        try (JsonWriter writer = context.jsonType().create(sbw);
             JsonWriter stat = JsonType.NORMAL.create(statSbw))
        {
            writer.startObject();
            stat.startObject();

            stat.key("failed");
            stat.startArray();
            writer.key("retry-suggest-types");
            writer.startArray();
            for (SuggestType type: failedSuggests) {
                writer.value(type.toString());
                stat.value(type.toString());
            }
            writer.endArray();
            stat.endArray();
            stat.key("ids");
            stat.startArray();
            writer.key("suggest");
            if (context.typeGroups().size() > 1) {
                writer.startObject();
                for (Map.Entry<Integer, List<List<SuggestItem>>> entry:
                    deduplicatedGrouped.entrySet())
                {
                    List<List<SuggestItem>> suggestGroup = entry.getValue();
                    Integer groupNr = entry.getKey();
                    writer.key(groupNr.toString());
                    writer.startArray();
                    List<SuggestItem> all = new ArrayList<>(suggestGroup.size() * context.length());
                    for (List<SuggestItem> items: suggestGroup) {
                        all.addAll(items);
                        printed += items.size();
                    }

                    Collections.sort(all);
                    printItemList(all, writer, stat);
                    writer.endArray();
                }
                writer.endObject();
            } else {
                writer.startArray();

                List<SuggestItem> all = new ArrayList<>(deduplicated.size() * context.length());
                for (List<SuggestItem> items: deduplicated) {
                    all.addAll(items);
                    printed += items.size();
                }

                Collections.sort(all);
                printItemList(all, writer, stat);

                writer.endArray();
            }

            writer.endObject();

            stat.endArray();
            stat.endObject();
        } catch (IOException e) {
            failed(e);
            return;
        }

        session.connection().setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Integer.toString(printed));
        session.connection().setSessionInfo(
            "result",
            statSbw.toString());

        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset())));
    }

    private void printItemList(
        final List<SuggestItem> items,
        final JsonWriter writer,
        final JsonWriter statWriter)
        throws IOException
    {
        for (SuggestItem item: items) {
            writer.startObject();
            writer.key("type");
            writer.value(item.type().toString());
            if (debug) {
                writer.key("score");
                writer.value(item.score());
            }

            writer.key("matches");
            writer.startObject();
            if (debug) {
                writer.key(DEBUG);
                writer.value(item.toString());
            }
            for (Map.Entry<String, String> textEntry
                : item.searchTexts())
            {
                List<String> matches = new ArrayList<>();
                for (String request: item.requests()) {
                    fillMatches(
                        matches,
                        textEntry.getValue(),
                        request);
                }
                if (matches.size() > 0) {
                    writer.key(textEntry.getKey());
                    writer.value(matches);
                }
            }
            writer.endObject();
            writer.key("id");
            writer.value(item.id());
            statWriter.value(item.id());
            for (Map.Entry<String, JsonObject> entry
                : item.payload().entrySet())
            {
//                        if (!entry.getKey().startsWith("#score")) {
                writer.key(entry.getKey());
                entry.getValue().writeValue(writer);
//                        }
            }
            writer.endObject();
        }
    }


}

