package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.request.util.BoostByOrderFieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class ChatsSuggestRule
    <T extends
        RequestProvider
        & SuggestRequestContextProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final String B_AND_B = ") AND (";
    private static final String SCORE = "#score";
    private static final String CHAT_DATA = "chat_data";
    private static final String DESCRIPTION_TOKENIZED =
        "chat_description_tokenized";
    //
    private static final String NAME = "chat_name";
    private static final String NAME_TOKENIZED = "chat_name_tokenized";
    private static final String DESCRIPTION = "chat_description";
    private static final String[] TOKENIZED_SEARCH_FIELDS = {
        DESCRIPTION_TOKENIZED,
        NAME_TOKENIZED
    };
    private static final String[] KEYWORD_SEARCH_FIELDS = {
        DESCRIPTION,
        NAME
    };
    private static final String[] MATCH_FIELDS = {
        DESCRIPTION,
        NAME
    };
    private static final BoostByOrderFieldsTermsSupplierFactory
        NAME_FIELD =
            new BoostByOrderFieldsTermsSupplierFactory(
                1f,
                Arrays.asList(TOKENIZED_SEARCH_FIELDS));
//    private static final BoostByOrderFieldsSwitchingTermsSupplierFactory
//        NAME_FIELD =
//            new BoostByOrderFieldsSwitchingTermsSupplierFactory(
//                1f,
//                1f,
//                Arrays.asList(TOKENIZED_SEARCH_FIELDS),
//                Arrays.asList(KEYWORD_SEARCH_FIELDS));
    private static final int MIN_LENGTH = 10;
    private static final List<String> KEYWORD_SEARCH_FIELDS_LIST =
        Arrays.asList(KEYWORD_SEARCH_FIELDS);
    private static final String CHAT_ID = "chat_id";
    private static final String RESOURCE_ID = CHAT_ID;

    private static final String[] SAVED_MESSAGES = {
        "сохраненные сообщения",
        "cохраненки",
        "избранное",
        "сообщения сохраненные",
        "favorites",
        "favorite",
        "saved messages",
        "messages saved",
        "messages"
    };
    private static final double SAVED_MESSAGES_SCORE = 100000.0;

    private final long failoverDelay;
    private final boolean localityShuffle;
    private final SuggestType suggestType;

    public ChatsSuggestRule(final Moxy moxy, final SuggestType suggestType) {
        this.suggestType = suggestType;
        failoverDelay = moxy.config().chatsSuggestFailoverDelay();
        localityShuffle = moxy.config().chatsSuggestLocalityShuffle();
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        String request = input.request();
        if (context.hadRequest(suggestType, request)) {
            context.logger().info("Request: " + request
                + "has already been executed. Skipping");
            callback.completed(Collections.emptyList());
            return;
        }
        execute(
            Math.max(MIN_LENGTH, context.length() << 1),
            new ChatsSuggestRequestContext(
                context,
                request,
                SearchRequestText.parseSuggest(input.request(), false),
                failoverDelay,
                localityShuffle,
                callback,
                suggestType));
    }

    public static void execute(
        final int requestedLength,
        final ChatsSuggestRequestContext context)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder("");
        SearchRequestText request = context.request;
        if (request.isEmptyIgnoreMentions() && request.text().length() > 0) {
            context.logger().info("No human words in  Request: " + request
                + " .Skipping ");
            context.callback.completed(Collections.emptyList());
            return;
        }
        if (request.text().length() < 2) {
            context.logger().info("Skipping search for short requests: "
                + request);
            context.callback.completed(Collections.emptyList());
            return;
        }
        if (context.channelsSearch) {
            sb.append("chat_id:1/*");
        }
        if (context.queryFilter != null) {
            sb.append('(');
            sb.append(context.queryFilter);
            sb.append(')');
        }
        if (request.hasWords()) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            sb.append('(');
            request.fieldsQuery(
                sb,
                NAME_FIELD,
                 B_AND_B);
            sb.append(')');
            if (request.singleWord()) {
                sb.append(" OR (");
                request.fieldsQuery(
                    sb,
                    KEYWORD_SEARCH_FIELDS_LIST,
                    B_AND_B,
                    2f);
                sb.append(')');
            }
        } else if (sb.length() == 0) {
            context.logger().info("Empty token list for Request: " + request
                + " .Skipping");
            context.callback.completed(Collections.emptyList());
            return;
        }

//        request.negationsQuery(sb, NAME_FIELD);
        LinkedHashSet<String> get = new LinkedHashSet<>();
        get.add(RESOURCE_ID);
        get.add(NAME);
        get.add(DESCRIPTION);
        get.add(SCORE);
        get.addAll(context.getFields);
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                + "&sync-searcher=false"
                + "&skip-nulls"
                + "&scorer=lucene&sort=%23score"
                + "&dp=fallback(chat_id+id)");
        User user = context.user();
        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        query.append("text", new String(sb));
        query.append("length", requestedLength);
        if (context.dps != null && context.dps.size() > 0) {
            for (String dp: context.dps) {
                query.append("dp", dp);
            }
        }
        if (context.postfilters != null && context.postfilters.size() > 0) {
            for (String pf: context.postfilters) {
                query.append("postfilter", pf);
            }
        }
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(context, requestedLength));
    }

    private static class ChatsSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final SearchRequestText request;
        private final FutureCallback<? super List<SuggestItem>> callback;
        private final User user;
        private final Set<String> getFields;
        private final String queryFilter;
        private final List<String> dps;
        private final List<String> postfilters;
        private final long failoverDelay;
        private final boolean localityShuffle;
        private final String requestUserId;
        private final boolean channelsSearch;
        private final SuggestType suggestType;

        // CSOFF: ParameterNumber
        ChatsSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final String requestString,
            final SearchRequestText request,
            final long failoverDelay,
            final boolean localityShuffle,
            final FutureCallback<? super List<SuggestItem>> callback,
            final SuggestType suggestType)
            throws BadRequestException
        {
            super(suggestRequestContext, requestString);
            this.request = request;
            this.callback = callback;
            this.suggestType = suggestType;
            this.channelsSearch = suggestType == SuggestType.CHANNELS;
            if (v2Org() != null && v2Org() != 0) {
                user = new User(
                    suggestRequestContext.proxy().config().v2ChatsService(),
                    new LongPrefix(v2Org()));
            } else {
                user = new User(
                    suggestRequestContext.proxy().config().chatsService(),
                    new LongPrefix(0));
            }
            getFields =
                session().params().get(
                    "chat_get",
                    new LinkedHashSet<>(),
                    new CollectionParser<>(
                        String::trim,
                        LinkedHashSet::new,
                        ','));
            if (channelsSearch) {
                queryFilter = session().params().getString("channel_filter", null);
                dps = session().params().getAll("channel_dp");
                postfilters = session().params().getAll("channel_postfilter");
            } else {
                queryFilter = session().params().getString("chat_filter", null);
                dps = session().params().getAll("chat_dp");
                postfilters = session().params().getAll("chat_postfilter");
            }
            this.failoverDelay = session().params().getLongDuration(
                "user_failover_delay",
                failoverDelay);
            this.localityShuffle = session().params().getBoolean(
                "user_locality_shuffle",
                localityShuffle);
            this.requestUserId = session().params().getString(
                "request-user-id",
                null);
        }
        // CSON: ParameterNumber

        @Override
        public User user() {
            return user;
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final ChatsSuggestRequestContext context;
        private final int requestedLength;

        Callback(
            final ChatsSuggestRequestContext context,
            final int requestedLength)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                System.err.println(
                    "Chats Lucene response: "
                    + JsonType.HUMAN_READABLE.toString(hits));
                context.proxy().filterChatResources(
                    context,
                    hits,
                    new FilteredResourcesCallback(
                        context,
                        requestedLength,
                        hits));
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class FilteredResourcesCallback
        extends AbstractFilterFutureCallback<Set<String>, List<SuggestItem>>
    {
        private final ChatsSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;

        FilteredResourcesCallback(
            final ChatsSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                if (!context.getFields.contains(field)) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                final String jsonString = doc.get(CHAT_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put(CHAT_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        private BasicSuggestItem generateSavedMessagesChat(
            final String matchedText)
            throws JsonException
        {
            JsonMap data = new JsonMap(BasicContainerFactory.INSTANCE);
            JsonList emptyList = new JsonList(BasicContainerFactory.INSTANCE);
            JsonList members = new JsonList(BasicContainerFactory.INSTANCE);
            members.add(new JsonString(context.requestUserId));
            JsonMap permissions = new JsonMap(BasicContainerFactory.INSTANCE);
            permissions.put("departments", emptyList);
            permissions.put("groups", emptyList);
            permissions.put("users", members);
            JsonList rights = new JsonList(BasicContainerFactory.INSTANCE);
            rights.add(new JsonString("read"));
            rights.add(new JsonString("write"));
            JsonMap roles = new JsonMap(BasicContainerFactory.INSTANCE);
            roles.put("admin", emptyList);
            String chatId = context.requestUserId + '_' + context.requestUserId;

            data.put(CHAT_ID, new JsonString(chatId));
            data.put("create_timestamp", new JsonDouble(0));
            data.put("exclude", emptyList);
            data.put("geo_type", new JsonString("chat"));
            data.put("members", members);
            data.put("moderation_status", new JsonString("ok"));
            data.put("permissions", permissions);
            data.put("private", JsonBoolean.TRUE);
            data.put("restriction_status", new JsonString("active"));
            data.put("rights", rights);
            data.put("roles", roles);
            data.put("version", new JsonLong(0));

            JsonMap doc = new JsonMap(BasicContainerFactory.INSTANCE);
            doc.put(CHAT_DATA, data);
            doc.put("id", new JsonString(chatId));
            filterGetFields(doc);

            return new BasicSuggestItem(
                chatId,
                context.suggestType,
                context.request.text(),
                Collections.singletonList(
                    new SimpleEntry<String, String>(NAME, matchedText)),
                SAVED_MESSAGES_SCORE,
                doc);
        }

        @Override
        public void completed(final Set<String> resources) {
            try {
                String request = context.request.text();
                int count = 0;
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));

                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    if (resourceId != null) {
                        ++count;
                        if (resources.contains(resourceId)) {
//                            String text = selectText(doc);
                            double score = doc.get(SCORE).asDouble();
                            List<Map.Entry<String, String>> searchTexts =
                                searchTexts(doc);
                            filterGetFields(doc);
                            jsonReformat(doc, jsonParser, consumer);
                            BasicSuggestItem item =
                                new BasicSuggestItem(
                                    resourceId,
                                    context.suggestType,
                                    request,
                                    searchTexts,
                                    score,
                                    doc);
                            items.add(item);
                        }
                    }
                }
                if (context.requestUserId != null && !context.channelsSearch) {
                    final String lowerCasedRequest =
                        context.request().toLowerCase(Locale.ROOT);
                    for (String saved: SAVED_MESSAGES) {
                        if (saved.startsWith(lowerCasedRequest)
                            || saved.contains(lowerCasedRequest))
                        {
                            items.add(generateSavedMessagesChat(saved));
                        }
                    }
                }
                if (items.size() >= context.length()
                    || count < requestedLength)
                {
                    callback.completed(items);
                } else {
                    execute(requestedLength << 1, context);
                }
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}

