package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class GroupsSuggestRule<T extends
                              RequestProvider
                              & SuggestRequestContextProvider>
    implements SearchRule<T, List<SuggestItem>> {
    private static final String SCORE = "#score";
    private static final String NAME = "org_group_name";
    //private static final String NAME_TOKENIZED = "org_department_name_tokenized_p";
    private static final int MIN_LENGTH = 10;
    //
    private static final String GROUP_DATA = "org_group_data";
    private static final String GROUP_ID = "org_group_id";

    private static final String[] SEARCH_FIELDS = {
        "org_group_name_p",
        "org_group_name_tokenized_p",
    };

    private static final String[] MATCH_FIELDS = {
        NAME
    };

    private final long failoverDelay;
    private final boolean localityShuffle;
    private final Moxy moxy;

    public GroupsSuggestRule(final Moxy moxy) {
        this.moxy = moxy;
        failoverDelay = moxy.config().usersSuggestFailoverDelay();
        localityShuffle = moxy.config().usersSuggestLocalityShuffle();
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException {
        SearchRequestText parsedRequest = new SearchRequestText(input.request());

        if (!parsedRequest.hasWords()) {
            callback.completed(Collections.emptyList());
            return;
        }

        GroupsSuggestContext context =
            new GroupsSuggestContext(
                input.suggestRequestContext(),
                input.request(),
                callback);

        StringBuilder sb =
            new StringBuilder("org_id_p:" + input.suggestRequestContext().v2Org());
        sb.append(" AND ");
        parsedRequest.fieldsQuery(sb, SEARCH_FIELDS);


        Set<String> get = new LinkedHashSet<>(context.getFields());
        get.add(GROUP_ID);
        get.add(GROUP_DATA);
        get.add(NAME);
        get.add(SCORE);

        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                    + "&sync-searcher=false"
                    + "&skip-nulls");

        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", context.user().prefix().toString());
        query.append("service", context.user().service());
        String searchText = new String(sb);
        query.append("text", searchText);
        query.append("db", "v2org");

        input.suggestRequestContext().logger().fine("GroupSuggestRule: request="
                                                        + input.request()
                                                        + ", backend_request: " + searchText);
        query.append("length", Math.max(MIN_LENGTH, input.suggestRequestContext().length() << 1));

        context.requestContext.proxy().sequentialRequest(
            context.requestContext.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            failoverDelay,
            localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.requestContext.contextGenerator(),
            new Callback(context));
    }

    private static class GroupsSuggestContext implements UniversalSearchProxyRequestContext {
        private final SuggestRequestContext requestContext;
        private final FutureCallback<? super List<SuggestItem>> callback;
        private final Set<String> getFields;
        private final String request;
        private final User user;

        public GroupsSuggestContext(
            final SuggestRequestContext requestContext,
            final String request,
            final FutureCallback<? super List<SuggestItem>> callback)
            throws BadRequestException {
            this.requestContext = requestContext;
            getFields =
                requestContext.session().params().get(
                    "group_get",
                    Collections.singleton(GROUP_DATA),
                    new CollectionParser<>(
                        String::trim,
                        LinkedHashSet::new,
                        ','));
            this.callback = callback;
            this.request = request;
            user = new User(
                requestContext.proxy().config().v2UsersService(),
                new LongPrefix(requestContext.v2Org()));
        }

        public Set<String> getFields() {
            return getFields;
        }

        public String request() {
            return request;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return requestContext.minPos();
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return requestContext.client();
        }

        @Override
        public Logger logger() {
            return requestContext.logger();
        }

        @Override
        public long lagTolerance() {
            return requestContext.lagTolerance();
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>> {
        private final GroupsSuggestContext context;
        private final int requestedLength;

        private Callback(
            final GroupsSuggestContext context) {
            super(context.callback);
            this.context = context;
            this.requestedLength = context.requestContext.length();
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                boolean leave = false;
                if (context.getFields().contains(field)) {
                    leave = true;
                }
                if (!leave) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer) {
            try {
                final String jsonString = doc.remove(GROUP_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put("group_data", obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field : MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new AbstractMap.SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));
                List<SuggestItem> items = new ArrayList<>(requestedLength);

                for (JsonObject hit : hits) {
                    JsonMap doc = hit.asMap();
                    String groupId = doc.getString(GROUP_ID);
                    double score = doc.getDouble(SCORE, 0.0);
                    List<Map.Entry<String, String>> searchTexts =
                        searchTexts(doc);
                    filterGetFields(doc);
                    jsonReformat(doc, jsonParser, consumer);

                    BasicSuggestItem item =
                        new BasicSuggestItem(
                            groupId,
                            SuggestType.GROUPS,
                            context.request(),
                            searchTexts,
                            score,
                            doc);
                    items.add(item);
                }

                callback.completed(items);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}
