package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserFields;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.UserSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class LoginsSuggestRule<T extends
        RequestProvider
        & SuggestRequestContextProvider>
        implements SearchRule<T, List<SuggestItem>>
{
    private static final String SCORE = "#score";
    private static final String NAME = "user_display_name";
    private static final String NICKNAME = "user_nickname";
    private static final String POSITION = "user_position";
    private static final String WEBSITE = "user_website";
    private static final String RESOURCE_ID = "user_id";
    private static final int MIN_LENGTH = 10;
    //
    private static final String CONTACT_NAME = "contact_name";
    private static final long MICROS_PER_SEC = 1000000;
    private static final String USER_DATA = "user_data";
    private static final String DISPLAY_NAME = "display_name";
    private static final String PASSPORT_DISPLAY_NAME = "passport_display_name";

    private static final String[] LOGIN_SEARCH_FIELDS = {
        UserFields.NICKNAME.global(),
    };

    private static final String[] MATCH_FIELDS = {
        UserFields.NICKNAME.stored()
    };

    private final long failoverDelay;
    private final boolean localityShuffle;
    private final Moxy moxy;
    private final User user;

    public LoginsSuggestRule(final Moxy moxy) {
        this.moxy = moxy;
        failoverDelay = moxy.config().usersSuggestFailoverDelay();
        localityShuffle = moxy.config().usersSuggestLocalityShuffle();
        this.user = new User(moxy.config().usersService(), new LongPrefix(0L));
    }

    @Override
    public void execute(
            final T input,
            final FutureCallback<? super List<SuggestItem>> callback)
            throws HttpException
    {
        // @ is restricted for login, onnly 360 logins have email like login
        int indexOfAt = input.request().indexOf('@');
        String normalized;
        if (indexOfAt < 0) {
            normalized = input.request().replaceAll("\\.", "-");
        } else {
            String realLogin = input.request().substring(0, indexOfAt).replaceAll("\\.", "-");
            String domain = input.request().substring(indexOfAt);
            normalized = realLogin + domain;
        }

        SearchRequestText parsedRequest = new SearchRequestText(normalized);

        if (!parsedRequest.hasWords()) {
            callback.completed(Collections.emptyList());
            return;
        }

        LoginsSuggestContext context =
            new LoginsSuggestContext(
                input.suggestRequestContext(),
                input.request(),
                callback);

        long userId = 0;
        StringBuilder sb =
            new StringBuilder("user_org_id:0");
        sb.append(" AND ");
        parsedRequest.fieldsQuery(sb, LOGIN_SEARCH_FIELDS);
        sb.append(" AND NOT ");
        sb.append(UserFields.SEARCH_PRIVACY.global());
        sb.append(":(");
        sb.append(SearchPrivacy.CONTACTS.value());
        sb.append(" OR ");
        sb.append(SearchPrivacy.NOBODY.value());
        sb.append(")");

        User user = new User(
                input.suggestRequestContext().proxy().config().usersService(),
                new LongPrefix(userId));
        Set<String> get = new LinkedHashSet<>(context.getFields());
        get.add(RESOURCE_ID);
        get.add(NAME);
        get.add(UserFields.NICKNAME.stored());
        get.add(POSITION);
        get.add(WEBSITE);
        get.add(SCORE);

        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                    + "&sync-searcher=false"
                    + "&skip-nulls");

        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        String searchText = new String(sb);
        query.append("text", searchText);

        input.suggestRequestContext().logger().fine("LoginSuggestRule: request="
                + input.request()
                + ", backend_request: " + searchText);
        query.append("length", Math.max(MIN_LENGTH, input.suggestRequestContext().length() << 1));

        context.requestContext.proxy().sequentialRequest(
            context.requestContext.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            failoverDelay,
            localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.requestContext.contextGenerator(),
            new Callback(context));
    }

    /*private static class LoginsSuggestCallback extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>> {

    }*/

    private class LoginsSuggestContext implements UniversalSearchProxyRequestContext {
        private final SuggestRequestContext requestContext;
        private final FutureCallback<? super List<SuggestItem>> callback;
        private final Set<String> getFields;
        private final String request;

        public LoginsSuggestContext(
            final SuggestRequestContext requestContext,
            final String request,
            final FutureCallback<? super List<SuggestItem>> callback)
            throws BadRequestException
        {
            this.requestContext = requestContext;
            getFields =
                requestContext.session().params().get(
                    "user_get",
                    new LinkedHashSet<>(),
                    new CollectionParser<>(
                        String::trim,
                        LinkedHashSet::new,
                        ','));
            this.callback = callback;
            this.request = request;
        }

        public Set<String> getFields() {
            return getFields;
        }

        public String request() {
            return request;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return requestContext.minPos();
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return requestContext.client();
        }

        @Override
        public Logger logger() {
            return requestContext.logger();
        }

        @Override
        public long lagTolerance() {
            return requestContext.lagTolerance();
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final LoginsSuggestContext context;
        private final int requestedLength;

        private Callback(
            final LoginsSuggestContext context)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = context.requestContext.length();
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                boolean leave = false;
                if (context.getFields().contains(field)) {
                    leave = true;
                }
                if (!leave) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                final String jsonString = doc.get(USER_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
//                    if (contactsSearch) {
                    final JsonObject contactName =
                        doc.get(CONTACT_NAME);
                    if (contactName != JsonNull.INSTANCE) {
                        JsonMap userData = obj.asMap();
                        userData.put(DISPLAY_NAME, contactName);
                        userData.put(PASSPORT_DISPLAY_NAME, contactName);
                    }
//                    }
                    doc.put(USER_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new AbstractMap.SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));
                List<SuggestItem> items = new ArrayList<>(requestedLength);

                for (JsonObject hit: hits) {
                    JsonMap doc = hit.asMap();
                    String userId = doc.getString("user_id");
                    long lastMsgTs =
                        doc.getLong(
                            "chat_last_message_timestamp",
                            0L) / MICROS_PER_SEC;
                    double score = doc.getDouble(SCORE, 0.0);
                    List<Map.Entry<String, String>> searchTexts =
                        searchTexts(doc);
                    jsonReformat(doc, jsonParser, consumer);
                    filterGetFields(doc);
                    UserSuggestItem item =
                        new UserSuggestItem(
                            userId,
                            SuggestType.LOGIN,
                            context.request(),
                            searchTexts,
                            score,
                            doc,
                            lastMsgTs);
                    items.add(item);
                }

                callback.completed(items);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}
