package ru.yandex.search.messenger.proxy.suggest.rules;

import java.nio.charset.CharacterCodingException;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.erratum.EmptyErratumResult;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ErratumResult;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestsProvider;

public class MisspellRule<
    T extends RequestsProvider & SuggestRequestContextProvider, U, R>
    implements SearchRule<T, R>
{
    private final ChainedSearchRule<T, U, List<String>, R> next;

    public MisspellRule(final ChainedSearchRule<T, U, List<String>, R> next) {
        this.next = next;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        List<String> requests = input.requests();
        ErratumClient client = context.proxy().misspellClient();
        ErratumCallback erratumCallback = new ErratumCallback(callback, input);
        if (client == null) {
            erratumCallback.completed(EmptyErratumResult.INSTANCE);
            return;
        }
        String request = null;
        if (requests != null && !requests.isEmpty()) {
            request = requests.get(0);
        }
        if (request != null) {
            request = SearchRequestText.normalize(request).trim();
        }
        if (request == null || request.isEmpty()) {
            erratumCallback.completed(EmptyErratumResult.INSTANCE);
            return;
        }
        if (context.debug()) {
            context.session().logger().info("Erratum request: " + request);
        }
        try {
            client.execute(
                request,
                context.session().listener().createContextGeneratorFor(client),
                new ErrorSuppressingFutureCallback<>(
                    erratumCallback,
                    EmptyErratumResult.INSTANCE));
        } catch (CharacterCodingException e) {
            throw new BadRequestException(
                "Failed to encode request '" + request + '\'',
                e);
        }
    }

    private class ErratumCallback
        extends AbstractFilterFutureCallback<ErratumResult, R>
    {
        private final T input;

        ErratumCallback(
            final FutureCallback<? super R> callback,
            final T input)
        {
            super(callback);
            this.input = input;
        }

        @Override
        public void completed(final ErratumResult erratumResult) {
            if (input.suggestRequestContext().debug()) {
                input.suggestRequestContext().session().logger().info(
                    "Erratum response: " + erratumResult);
            }
            try {
                List<String> requests = new ArrayList<>(input.requests());
                if (erratumResult.code() == ErratumResult.CODE_CORRECTED) {
                    requests.add(erratumResult.text());
                }
                next.execute(input, requests, callback);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }
}

