package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.MetaStoringUsersFilter;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.UsersFilter;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextRequestAllFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestUsersFilterProvider;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.request.util.BoostByOrderFieldsTermsSupplierFactory;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class ResolveContactsUsersRule<
    T extends
        RequestProvider
        & SuggestRequestContextProvider
        & SuggestUsersFilterProvider
        & SuggestChatsFilterProvider,
    U extends SuggestRequestContextRequestAllFilterProvider,
    R>
    implements SearchRule<T, R>
{
    public static final String REQUEST_USER_ID =
        "request-user-id";
    private static final MetaStoringUsersFilter EMPTY_USERS_FILTER =
        new MetaStoringUsersFilter(Collections.emptyMap());
    private static final float KEYWORD_BOOST = 5f;
    private static final int MIN_LENGTH = 10;
    private static final String CONTACT_ID = "contact_id";
    private static final String SCORE = "#score";
    private static final String B_AND_B = ") AND (";
    private static final String AND = " AND ";
    private static final String OR_B = " OR (";
    private static final String OR = " OR ";
    private static final String NAME_TOKENIZED_P =
        "contact_name_tokenized_p";
    private static final String NAME = "contact_name";
    private static final String NAME_P = "contact_name_p";

    private static final String[] LOGIN_FIELDS = {
        NAME_TOKENIZED_P,
    };
    private static final String[] LOGIN_FIELDS_KEYWORD = {
        NAME_P,
    };
    private static final List<String> MENTION_FIELDS =
        Arrays.asList(LOGIN_FIELDS);
    private static final List<String> MENTION_FIELDS_KEYWORD_LIST =
        Arrays.asList(LOGIN_FIELDS_KEYWORD);

    private static final String[] SEARCH_FIELDS_TOKENIZED = {
        NAME_TOKENIZED_P
    };
    private static final String[] SEARCH_FIELDS_KEYWORD = {
        NAME_P
    };

    private static final BoostByOrderFieldsTermsSupplierFactory
        NAME_FIELD =
            new BoostByOrderFieldsTermsSupplierFactory(
                1f,
                Arrays.asList(SEARCH_FIELDS_TOKENIZED));
    private static final List<String> SEARCH_FIELDS_KEYWORD_LIST =
        Arrays.asList(SEARCH_FIELDS_KEYWORD);

    private final ChainedSearchRule<T, U, UsersFilter, R> next;
    private final long failoverDelay;
    private final boolean localityShuffle;
//    private final Moxy moxy;

    public ResolveContactsUsersRule(
        final Moxy moxy,
        final ChainedSearchRule<T, U, UsersFilter, R> next)
    {
//        this.moxy = moxy;
        this.next = next;
        failoverDelay = moxy.config().usersSuggestFailoverDelay();
        localityShuffle = moxy.config().usersSuggestLocalityShuffle();
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        String request = input.request();
        if (context.session().params().getString(REQUEST_USER_ID, null)
            == null)
        {
            context.logger().info("Empty reqiest-user-id "
                + "Skipping contacts suggest");
            next.execute(
                input,
                EMPTY_USERS_FILTER,
                callback);
            return;
        }
        if (context.hadRequest(
            SuggestType.CONTACTS,
            "ContacsFilter_" + request))
        {
            context.logger().info("Request: " + request
                + " has already been executed. Skipping");
            next.execute(
                input,
                EMPTY_USERS_FILTER,
                callback);
            return;
        }
        execute(
            input,
            Math.max(MIN_LENGTH, context.length() << 1),
            new ContactsSuggestRequestContext(
                context,
                request,
                SearchRequestText.parseSuggest(input.request(), false),
                failoverDelay,
                localityShuffle),
            callback);
    }

    //CSOFF: ParameterNumber
    public void execute(
        final T input,
        final int requestedLength,
        final ContactsSuggestRequestContext context,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder("");
        SearchRequestText request = context.request;
        if (context.queryFilter != null) {
            sb.append('(');
            sb.append(context.queryFilter);
            sb.append(')');
        }
        boolean closeBrace = false;
        if (request.hasMentions() || request.hasWords()) {
            if (sb.length() > 0) {
                sb.append(AND);
            }
            sb.append('(');
            closeBrace = true;
        }
        if (request.hasMentions()) {
            sb.append('(');
            request.mentionsQuery(
                sb,
                new FieldsTermsSupplierFactory(MENTION_FIELDS),
                B_AND_B);
            sb.append(')');
            if (request.singleMentionWord()) {
                System.err.println("Single mention word");
                sb.append(OR_B);
                request.mentionsQuery(
                    sb,
                    MENTION_FIELDS_KEYWORD_LIST,
                    B_AND_B,
                    KEYWORD_BOOST);
                sb.append(')');
            }
        }
        if (request.hasWords()) {
            if (request.hasMentions()) {
                sb.append(OR);
            }
            sb.append('(');
            request.fieldsQuery(
                sb,
                NAME_FIELD,
                 B_AND_B);
            sb.append(')');
            System.err.println("Pre Single word");
            if (request.singleWord()) {
                System.err.println("Single word");
                sb.append(OR_B);
                request.fieldsQuery(
                    sb,
                    SEARCH_FIELDS_KEYWORD_LIST,
                    B_AND_B,
                    KEYWORD_BOOST);
                sb.append(')');
            }
        }
        if (closeBrace) {
            sb.append(')');
        }
        if (sb.length() == 0) {
            context.logger().info("Empty token list for Request: " + request
                + " .Skipping");
            next.execute(
                input,
                EMPTY_USERS_FILTER,
                callback);
            return;
        }
//        request.negationsQuery(sb, NAME_FIELD);
        LinkedHashSet<String> get = new LinkedHashSet<>();
        get.add(CONTACT_ID);
        get.add(NAME);
        get.add(SCORE);
        get.addAll(context.getFields);
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                    + "&sync-searcher=false"
                    + "&skip-nulls"
                    + "&scorer=lucene&sort=multi(%23score,contact_name)"
                    + "&dp=fallback(contact_id+id)");
        User user = context.user();
        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        query.append("text", new String(sb));
        query.append("length", requestedLength);
        if (context.dps != null && context.dps.size() > 0) {
            for (String dp: context.dps) {
                query.append("dp", dp);
            }
        }
        if (context.postfilters != null && context.postfilters.size() > 0) {
            for (String pf: context.postfilters) {
                query.append("postfilter", pf);
            }
        }
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(callback, input));
    }
    //CSON: ParameterNumber

    private static class ContactsSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final SearchRequestText request;
//        private final String type;
        private final User user;
        private final Set<String> getFields;
        private final String queryFilter;
        private final List<String> dps;
        private final List<String> postfilters;
        private final long failoverDelay;
        private final boolean localityShuffle;

        // CSOFF: ParameterNumber
        ContactsSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final String requestString,
            final SearchRequestText request,
            final long failoverDelay,
            final boolean localityShuffle)
            throws BadRequestException
        {
            super(suggestRequestContext, requestString);
            this.request = request;
            String userId = session().params().getString(REQUEST_USER_ID);
            user = new User(
                suggestRequestContext.proxy().config().messagesService(),
                new StringPrefix(userId));
            getFields =
                session().params().get(
                    "contact_get",
                    new LinkedHashSet<>(),
                    new CollectionParser<>(
                        String::trim,
                        LinkedHashSet::new,
                        ','));
            queryFilter = session().params().getString("contact_filter", null);
            dps = session().params().getAll("contact_dp");
            postfilters = session().params().getAll("contact_postfilter");
            this.failoverDelay = session().params().getLongDuration(
                "contact_failover_delay",
                failoverDelay);
            this.localityShuffle = session().params().getBoolean(
                "contact_locality_shuffle",
                localityShuffle);
        }
        // CSON: ParameterNumber

        @Override
        public User user() {
            return user;
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, R>
    {
        private final T input;

        Callback(
            final FutureCallback<? super R> callback,
            final T input)
        {
            super(callback);
            this.input = input;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (input.suggestRequestContext().debug()) {
                    input.suggestRequestContext().session().logger().info(
                        "ResolveContactsMembersRule Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                Map<String, JsonMap> usersSet = null;
                if (hits.size() == 0) {
                    usersSet = Collections.emptyMap();
                } else {
                    for (JsonObject jo: hits) {
                        JsonMap hit = jo.asMap();
                        String id = hit.getOrNull(CONTACT_ID);
                        if (id != null && input.usersFilter().testUser(id)) {
                            if (usersSet == null) {
                                usersSet = new HashMap<>();
                            }
                            usersSet.put(id, hit);
                        }
                    }
                    if (usersSet == null) {
                        usersSet = Collections.emptyMap();
                    }
                }
                next.execute(
                    input,
                    new MetaStoringUsersFilter(usersSet),
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }
}

