package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.MetaStoringUsersFilter;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.UsersFilter;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

@SuppressWarnings("StringSplitter")
public class ResolvePVPUsersRule<
    T extends RequestProvider & SuggestRequestContextProvider,
    U,
    R>
    implements SearchRule<T, R>
{
    private static final String CHAT_MEMBERS = "chat_members";
    private static final String GET_FIELDS = CHAT_MEMBERS
        + ",chat_last_message_timestamp,chat_message_count";

    private final ChainedSearchRule<T, U, UsersFilter, R> next;

    public ResolvePVPUsersRule(
        final ChainedSearchRule<T, U, UsersFilter, R> next)
    {
        this.next = next;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        ResolveChatsSuggestRequestContext context =
            new ResolveChatsSuggestRequestContext(
                input.suggestRequestContext(),
                input.request());
        ProxySession session = context.session();
        final String userId =
            session.params().getString("pvp-user-id");
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar&sync-searcher=false");
        query.append("get", GET_FIELDS);
        query.append("prefix", context.user().prefix().toString());
        query.append("service", context.user().service());
        query.append("text", "chat_members:" + userId);
        query.append("dp", "contains(chat_id,/ slash)");
        query.append("postfilter", "slash != 1");
        query.append("outer", "deduplicate:chat_members");
        query.append("length", "200");
        query.append("sort", "chat_last_message_timestamp");
        /**
         Parallel should not be used, when parallel index requests
         in consumer are enabled
         **/
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay(),
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(callback, input));
    }

    private static class ResolveChatsSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final User user;
        private final long failoverDelay;

        ResolveChatsSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final String requestString)
            throws BadRequestException
        {
            super(suggestRequestContext, requestString);
            if (v2Org() != null && v2Org() != 0) {
                user = new User(
                    suggestRequestContext.proxy().config().v2ChatsService(),
                    new LongPrefix(v2Org()));
            } else {
                user = new User(
                    suggestRequestContext.proxy().config().chatsService(),
                    new LongPrefix(0));
            }
            this.failoverDelay = suggestRequestContext.session().params().getLong(
                "failover-delay",
                proxy().config().messagesSuggestFailoverDelay());
        }

        public long failoverDelay() {
            return failoverDelay;
        }

        @Override
        public User user() {
            return user;
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, R>
    {
        private final T input;

        Callback(
            final FutureCallback<? super R> callback,
            final T input)
        {
            super(callback);
            this.input = input;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (input.suggestRequestContext().debug()) {
                    input.suggestRequestContext().session().logger().info(
                        "ResolveChatMembersRule Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                Map<String, JsonMap> usersSet = null;
                if (hits.size() == 0) {
                    usersSet = Collections.emptyMap();
                } else {
                    for (JsonObject jo: hits) {
                        JsonMap hit = jo.asMap();
                        String members = hit.getOrNull(CHAT_MEMBERS);
                        if (members != null) {
                            if (usersSet == null) {
                                usersSet = new HashMap<>();
                            }
                            hit.remove(CHAT_MEMBERS);
                            for (String member: members.split("\n")) {
                                usersSet.put(member, hit);
                            }
                        }
                    }
                    if (usersSet == null) {
                        usersSet = Collections.emptyMap();
                    }
                }
                next.execute(
                    input,
                    new MetaStoringUsersFilter(usersSet),
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }
}

