package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.ChatFields;
import ru.yandex.search.messenger.UserChats;
import ru.yandex.search.messenger.UserClearedChats;
import ru.yandex.search.messenger.proxy.suggest.AcceptAllChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.AcceptNoneChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.BasicChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.ChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.rules.pure.ChainedSearchRule;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

@SuppressWarnings("StringSplitter")
public class ResolveUserChatsRule<
    T extends RequestProvider & SuggestRequestContextProvider,
    U,
    R>
    implements SearchRule<T, R>
{
    private static final String CHAT_ID = "chat_id";
    private static final String GET_FIELDS =
        ChatFields.CHAT_ID.stored() + ","
            + UserChats.CHATS.stored() + ","
            + UserFields.CLEARED_CHATS.stored();

    private static final int MAX_CHATS_RESOLVE_COUNT = 2100;

    private final ChainedSearchRule<T, U, ChatsFilter, R> next;

    public ResolveUserChatsRule(
        final ChainedSearchRule<T, U, ChatsFilter, R> next)
    {
        this.next = next;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super R> callback)
        throws HttpException
    {
        ResolveUserChatsSuggestRequestContext context =
            new ResolveUserChatsSuggestRequestContext(
                input.suggestRequestContext(),
                input.request());
        ProxySession session = context.session();
        if (session.params().getBoolean("message-global-search", false)) {
            next.execute(input, AcceptAllChatsFilter.INSTANCE, callback);
            return;
        }
        final String chatId =
            session.params().getString("chat-id-filter", null);
        final String userId =
            session.params().getString("user-id", null);
        if (chatId != null) {
            // TODO also get pvp chats
            if (userId == null) {
                session.logger().warning("chat-id-filter but user-id is empty "
                    + "can not search without cleared no message-global-search params: will return empty"
                    + " results");
                next.execute(input, AcceptNoneChatsFilter.INSTANCE, callback);
            } else {
                StringBuilder textSb = new StringBuilder("id:");
                textSb.append(ru.yandex.search.messenger.UserFields.id(userId, "0"));
                QueryConstructor query =
                    new QueryConstructor(
                        "/search?IO_PRIO=0&json-type=dollar&sync-searcher=false");
                query.append("get", UserFields.CLEARED_CHATS.stored());
                query.append("prefix", context.user().prefix().toString());
                query.append("service", context.user().service());
                query.append("text", textSb.toString());
                if (context.v2Org() != null && context.v2Org() != 0) {
                    query.append("db", "v2org");
                }

                /**
                 Parallel should not be used, when parallel index requests
                 in consumer are enabled
                 **/
                context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    context.failoverDelay(),
                    true,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new SingleChatCallback(callback, input, chatId, context));
            }
        } else {
            if (userId == null) {
                session.logger().warning("Empty chat-id-filter, user-id, "
                    + "no message-global-search params: will return empty"
                    + " results");
                next.execute(input, AcceptNoneChatsFilter.INSTANCE, callback);
                return;
            }
            session.logger().info("Resolving chats for user-id: " + userId);
            StringBuilder textSb = new StringBuilder();
            if (context.namespaces() != null) {
                textSb.append('(');
                if (context.namespaces().isEmpty() && context.namespacesGuids().isEmpty()) {
                    session.logger().warning("NO namespaces and no guids, skipping messages suggest");
                    next.execute(input, AcceptNoneChatsFilter.INSTANCE, callback);
                    return;
                }
                if (!context.namespaces().isEmpty()) {
                    textSb.append("(");
                    textSb.append(ru.yandex.ps.search.messenger.ChatFields.NAMESPACE.global());
                    textSb.append(":(");
                    for (String namespace: context.namespaces()) {
                        textSb.append(namespace);
                        textSb.append(' ');
                    }
                    textSb.setLength(textSb.length() - 1);
                    textSb.append(") AND ");
                    if (context.v2Org() != null) {
                        textSb.append("chat_members_p");
                    } else {
                        textSb.append(ChatFields.MEMBERS.global());
                    }

                    textSb.append(":");
                    textSb.append(userId);
                    textSb.append(")");
                }

                if (!context.namespacesGuids().isEmpty()) {
                    if (!context.namespaces().isEmpty()) {
                        textSb.append(" OR ");
                    }

                    textSb.append("id");
                    textSb.append(":(");
                    for (String guid: context.namespacesGuids()) {
                        textSb.append("chat_");
                        textSb.append(userId);
                        textSb.append('_');
                        textSb.append(guid);
                        textSb.append(' ');
                        textSb.append("chat_");
                        textSb.append(guid);
                        textSb.append('_');
                        textSb.append(userId);
                        textSb.append(' ');
                    }
                    textSb.setLength(textSb.length() - 1);
                    textSb.append(')');
                }

                textSb.append(')');
            } else {
                if (context.v2Org() != null) {
                    textSb.append("chat_members_p");
                } else {
                    textSb.append(ChatFields.MEMBERS.global());
                }
                textSb.append(":");
                textSb.append(userId);
            }

            if (context.v2Org() == null) {
                if (textSb.length() > 0) {
                    textSb.append(" OR (");
                } else {
                    textSb.append("(");
                }

                textSb.append("id:");
                textSb.append(UserChats.id(userId));
                textSb.append(" OR id:");
                textSb.append(ru.yandex.search.messenger.UserFields.id(userId, "0"));
                textSb.append(")");
            }

            QueryConstructor query =
                new QueryConstructor(
                    "/search?IO_PRIO=0&json-type=dollar&sync-searcher=false");
            query.append("get", GET_FIELDS);
            query.append("prefix", context.user().prefix().toString());
            query.append("service", context.user().service());
            query.append("text", textSb.toString());
            query.append("sort", ChatFields.LAST_MESSAGE_TIMESTAMP.stored());
            if (context.v2Org() != null) {
                query.append("db", "v2org");
            }
            //query.append("group", CHAT_ID);
            query.append("length", MAX_CHATS_RESOLVE_COUNT);
            /**
             Parallel should not be used, when parallel index requests
             in consumer are enabled
             **/
            context.proxy().sequentialRequest(
                context.session(),
                context,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                context.failoverDelay(),
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                new Callback(callback, input, context));
        }
    }

    private static class ResolveUserChatsSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final User user;
        private final boolean debug;
        private final boolean pvpChats;
        private final boolean groupChats;
        private final long failoverDelay;

        ResolveUserChatsSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final String requestString)
            throws BadRequestException
        {
            super(suggestRequestContext, requestString);
            groupChats = suggestRequestContext.session().params().getBoolean("messages-group-chats", true);
            pvpChats = suggestRequestContext.session().params().getBoolean("messages-pvp-chats", true);
            if (v2Org() != null && v2Org() != 0) {
                user = new User(
                    suggestRequestContext.proxy().config().v2ChatsService(),
                    new LongPrefix(v2Org()));
            } else {
                user = new User(
                    suggestRequestContext.proxy().config().chatsService(),
                    new LongPrefix(0));
            }

            debug = suggestRequestContext.session().params().getBoolean(
                "resolve-chats-debug",
                false);
            this.failoverDelay = suggestRequestContext.session().params().getLong(
                "failover-delay",
                proxy().config().messagesSuggestFailoverDelay());
        }

        public boolean pvpChats() {
            return pvpChats;
        }

        public boolean groupChats() {
            return groupChats;
        }

        public long failoverDelay() {
            return failoverDelay;
        }

        @Override
        public User user() {
            return user;
        }
    }

    private class SingleChatCallback
        extends AbstractFilterFutureCallback<JsonObject, R>
    {
        private final T input;
        private final String chatId;
        private final ResolveUserChatsSuggestRequestContext context;

        SingleChatCallback(
            final FutureCallback<? super R> callback,
            final T input,
            final String chatId,
            final ResolveUserChatsSuggestRequestContext context)
        {
            super(callback);
            this.input = input;
            this.context = context;
            this.chatId = chatId;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (context.debug) {
                    context.session().logger().fine(
                        "ResolveUserChatMembersRule Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }
                Map<String, Long> clearedChats = null;
                Map<String, Long> chats =
                    Collections.singletonMap(chatId, UserClearedChats.DEFAULT_VALUE);

                for (JsonObject item: hits) {
                    JsonMap hit = item.asMap();
                    clearedChats
                        = hit.get(UserFields.CLEARED_CHATS.stored(),
                        null,
                        UserClearedChats.PARSER);
                }

                if (clearedChats != null && clearedChats.size() > 0) {
                    Long ts = clearedChats.get(chatId);
                    if (ts != null) {
                        chats = Collections.singletonMap(chatId, ts);
                    }
                }

                BasicChatsFilter filter = new BasicChatsFilter(chats);
                if (context.debug) {
                    context.session().logger().fine(
                        "ChatsFilter.set: " + filter.chats());
                }
                next.execute(
                    input,
                    filter,
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, R>
    {
        private final T input;
        private final ResolveUserChatsSuggestRequestContext context;

        Callback(
            final FutureCallback<? super R> callback,
            final T input,
            final ResolveUserChatsSuggestRequestContext context)
        {
            super(callback);
            this.input = input;
            this.context = context;
        }


        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                if (context.debug) {
                    context.session().logger().fine(
                        "ResolveUserChatMembersRule Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }
                Map<String, Long> chatsSet;
                if (hits.size() == 0) {
                    chatsSet = null;
                } else {
                    boolean nameSpaceFilter = context.namespaces() != null && !context.namespaces().isEmpty();

                    Map<String, Long> clearedChats = new LinkedHashMap<>();
                    chatsSet = new LinkedHashMap<>();
                    for (JsonObject item: hits) {
                        JsonMap hit = item.asMap();
                        Set<String> userChats =
                            hit.get(
                                UserChats.CHATS.stored(),
                                null,
                                UserChats.CHATS_SET_PARSER);
                        if (userChats != null) {
                            Long defaultTs = -1L;
                            for (String userChat: userChats) {
                                if (nameSpaceFilter) {
                                    int firstIndex = userChat.indexOf('/');
                                    int secondIndex;
                                    if (firstIndex > 0 && firstIndex < userChat.length() - 1) {
                                        secondIndex = userChat.indexOf('/', firstIndex + 1);
                                    } else {
                                        secondIndex = -1;
                                    }
                                    if (firstIndex <= 0
                                        || secondIndex <= firstIndex
                                        || !context.namespaces().contains(
                                            userChat.substring(firstIndex + 1, secondIndex)))
                                    {
                                        context.logger().info("Namespace filtering chat " + userChat);
                                        continue;
                                    }
                                }
                                chatsSet.put(userChat, defaultTs);
                            }
                        }

                        Map<String, Long> cleared
                            = hit.get(UserFields.CLEARED_CHATS.stored(),
                            null,
                            UserClearedChats.PARSER);
                        if (cleared != null) {
                            clearedChats.putAll(cleared);
                        }

                        String chatId = hit.getOrNull(CHAT_ID);
                        if (chatId != null) {
                            chatsSet.put(chatId, UserClearedChats.DEFAULT_VALUE);
                        }
                    }

                    if (clearedChats.size() > 0) {
                        for (Map.Entry<String, Long> entry
                            : clearedChats.entrySet())
                        {
                            chatsSet.replace(entry.getKey(), entry.getValue());
                        }
                    }
                }
                ChatsFilter filter;
                if (chatsSet == null) {
                    filter = AcceptNoneChatsFilter.INSTANCE;
                } else {
                    context.session().logger()
                        .info("Resolved chats filter size: " + chatsSet + " " + this.getClass());
                    filter = new BasicChatsFilter(chatsSet);
                    if (context.debug) {
                        context.session().logger().fine(
                            "ChatsFilter.set: " + filter.chats());
                    }
                }
                next.execute(
                    input,
                    filter,
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }
}
