package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Supplier;

import ru.yandex.search.messenger.proxy.Synonyms;
import ru.yandex.search.request.util.Term;
import ru.yandex.util.string.StringUtils;

public class SynonymsFieldsTermsSupplier implements Supplier<Term> {
    private static final int MAX_SYNONYMS = 20;
    private static final int MIN_SUGGEST_CHARS = 2;

    private final Supplier<String> fields;
    private final String token;
    private float boost;
    private final Synonyms synonyms;

    //CSOFF: ParameterNumber
    public SynonymsFieldsTermsSupplier(
        final Supplier<String> fields,
        final String token,
        final float initialBoost,
        final Synonyms synonyms)
    {
        this.fields = fields;
        this.token = token;
        this.boost = initialBoost;
        this.synonyms = synonyms;
    }
    //CSON: ParameterNumber

    @Override
    public Term get() {
        String field = fields.get();
        if (field == null) {
            return null;
        } else {
            String suggestStr;
            boolean finishedWord = false;
            if (token.endsWith("*")) {
                suggestStr = token.substring(0, token.length() - 1);
            } else {
                suggestStr = token;
                finishedWord = true;
            }
            List<String> synonyms;
            if (suggestStr.length() >= MIN_SUGGEST_CHARS) {
                synonyms =
                    this.synonyms.suggest(suggestStr, true);
                if (synonyms.size() > MAX_SYNONYMS) {
                    synonyms = synonyms.subList(0, MAX_SYNONYMS);
                }
            } else {
                synonyms = new ArrayList<>();
            }
            if (finishedWord
                && token.length() > 1
                && isCyrillic(token.charAt(token.length() - 1)))
            {
                synonyms.add(token + 'ь');
                synonyms.add(token + 'ъ');
            }
            Term term = new OrTerm(field, token, synonyms, boost);
            boost *= 2;
            return term;
        }

    }

    private boolean isCyrillic(final char c) {
        return (c >= 0x400 && c <= 0x465)
            || (c >= 0x48a && c <= 0x52f);
    }

    private static class OrTerm extends Term {
        private final Collection<String> synonyms;

        OrTerm(
            final String field,
            final String token,
            final Collection<String> synonyms)
        {
            super(field, token);
            this.synonyms = synonyms;
        }

        //CSOFF: ParameterNumber
        OrTerm(
            final String field,
            final String token,
            final Collection<String> synonyms,
            final float boost)
        {
            super(field, token, boost);
            this.synonyms = synonyms;
        }
        //CSON: ParameterNumber

        @Override
        public void appendTo(final StringBuilder sb) {
            sb.append(field);
            sb.append(':');
            if (synonyms != null && synonyms.size() > 0) {
                sb.append('(');
                sb.append(token);
                if (boost != 0f) {
                    sb.append('^');
                    sb.append(boost);
                }
                sb.append(" ");
                sb.append(StringUtils.join(synonyms, ' '));
                sb.append(')');
            } else {
                sb.append(token);
                if (boost != 0f) {
                    sb.append('^');
                    sb.append(boost);
                }
            }
        }
    }
}
