package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.ps.search.messenger.UserFields;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserClearedChats;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.UserSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.users.AbstractUserSuggestRule;
import ru.yandex.search.messenger.proxy.suggest.rules.users.UsersSuggestRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

public class UsersPvp2Rule<T extends RequestProvider & SuggestRequestContextProvider,
    U,
    R> extends AbstractUserSuggestRule
    implements SearchRule<T, List<SuggestItem>>
{
    private static final String GET_FIELDS =
        "chat_id,chat_members,chat_last_message_timestamp,chat_message_count,"
            + "user_id,#score,"
            + UserFields.SEARCH_PRIVACY.stored();

    private static final float KEYWORD_BOOST = 5f;
    //
    private static final String CONTACT_NAME = "contact_name";

    private static final String CONTACT_ID = "contact_id";
    //
    private static final String DISPLAY_NAME = "display_name";
    private static final String PASSPORT_DISPLAY_NAME = "passport_display_name";

    public UsersPvp2Rule(final Moxy moxy) {
        super(moxy);
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SearchRequestText requestText =
            SearchRequestText.parseSuggest(input.request().replaceAll(",", " "), false);
        UsersSuggestRequestContext context =
            new UsersSuggestRequestContext(
                input.suggestRequestContext(),
                input.request(),
                requestText,
                null,
                null,
                true,
                moxy.config().usersSuggestFailoverDelay(),
                moxy.config().usersSuggestLocalityShuffle(),
                SuggestType.USERS_PVP,
                callback,
                false);
        ProxySession session = context.session();
        final String userId =
            session.params().getString("pvp-user-id");
        int requestSize = context.length();
        String queryFilter;
        String userFilter = session.params().getString("pvp_user_filter", null);
        if (context.v2Org() != null && context.v2Org() != 0 && userFilter != null) {
            userFilter = userFilter.replaceFirst("user_org_id:0", "user_org_id:" + context.v2Org());
        }

        queryFilter = userFilter;

        Set<String> getFields =
            session.params().get(
                "user_get",
                new LinkedHashSet<>(),
                new CollectionParser<>(
                    String::trim,
                    LinkedHashSet::new,
                    ','));

        StringBuilder sb = new StringBuilder();
        if (queryFilter != null) {
            sb.append('(');
            sb.append(queryFilter);
            sb.append(')');
        }
        if (requestText.hasWords()) {
            if (queryFilter != null) {
                sb.append(" AND (");
            } else {
                sb.append("(");
            }
            requestText.fieldsQuery(
                sb,
                searchFields(2),
                B_AND_B);
            if (requestText.singleWord()) {
                sb.append(OR_B);
                requestText.fieldsQuery(
                    sb,
                    keywordFields(2),
                    B_AND_B,
                    KEYWORD_BOOST);
                sb.append(')');
            }
            sb.append(')');
            if (queryFilter != null) {
                //sb.append(')');
            }
        }

        if (sb.length() <= 0) {
            sb.append("type:user");
        }
        sb.append(" AND NOT ");
        sb.append(UserFields.SEARCH_PRIVACY.global());
        sb.append(':');
        sb.append(SearchPrivacy.NOBODY.value());

        StringBuilder leftJoin = new StringBuilder();
        leftJoin.append("left_join(other_guid,user_id,");
        leftJoin.append(sb);
        for (String getField: getFields) {
            leftJoin.append(',');
            leftJoin.append(getField);
            leftJoin.append(' ');
            leftJoin.append(getField);
        }
        leftJoin.append(",user_id user_id");
        leftJoin.append(')');

        StringBuilder totalGet = new StringBuilder();
        totalGet.append(GET_FIELDS);
        if (getFields.size() > 0) {
            totalGet.append(',');
            for (String getField: getFields) {
                totalGet.append(getField);
                totalGet.append(',');
            }
            totalGet.setLength(totalGet.length() - 1);
        }

        StringBuilder textSb = new StringBuilder();
        if (context.v2Org() != null) {
            textSb.append("chat_members_p:");
        } else {
            textSb.append("chat_members:");
        }

        textSb.append(userId);
        if (context.namespaces() != null) {
            if (context.namespacesGuids().isEmpty() && context.namespaces().isEmpty()) {
                context.callback().completed(Collections.emptyList());
                context.logger().info("Empty namespace guids and empty namespaces");
                return;
            }

            textSb.append(" AND (");
            if (!context.namespaces().isEmpty()) {
                textSb.append(ChatFields.NAMESPACE.global());
                textSb.append(":(");
                for (String namespace: context.namespaces()) {
                    textSb.append(namespace);
                    textSb.append(' ');
                }
                textSb.setLength(textSb.length() - 1);
                textSb.append(")");
            }

            if (!context.namespacesGuids().isEmpty()) {
                if (!context.namespaces().isEmpty()) {
                    textSb.append(" OR ");
                }

                textSb.append("id");
                textSb.append(":(");
                for (String guid: context.namespacesGuids()) {
                    textSb.append("chat_");
                    textSb.append(context.userGuid());
                    textSb.append('_');
                    textSb.append(guid);
                    textSb.append(' ');
                    textSb.append("chat_");
                    textSb.append(guid);
                    textSb.append('_');
                    textSb.append(context.userGuid());
                    textSb.append(' ');
                }
                textSb.setLength(textSb.length() - 1);
                textSb.append(')');
            }

            textSb.append(")");
        }

        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar&sync-searcher=false");
        query.append("get", totalGet.toString());
        query.append("prefix", context.user().prefix().toString());
        query.append("service", context.user().service());
        query.append("text", textSb.toString());
        query.append("dp", "keep_pvp_users(" + userId + " other_guid)");
        query.append("dp", leftJoin.toString());
        query.append("experimental", "true");
        query.append("length", context.length());
        query.append("collector", "sorted");
        query.append("keep-right-null", "false");
        query.append("sort", "chat_last_message_timestamp");
        query.append("left_dp_score", "#score");
        if (context.database() != null) {
            query.append("db", context.database());
        }
        /**
         Parallel should not be used, when parallel index requests
         in consumer are enabled
         **/
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay(),
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(context, context.length()));
    }

    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;

        Callback(
            final UsersSuggestRequestContext context,
            final int requestedLength)
        {
            super(context.callback());
            this.context = context;
            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                if (context.debug()) {
                    context.session().logger().info(
                        "Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                ExtractUserDataCallback callback =
                    new ExtractUserDataCallback(
                        context,
                        requestedLength,
                        hits);

                String get = UserFields.CLEARED_CHATS.stored()
                    + ',' + UserFields.BLACKLISTED_USERS.stored();
                QueryConstructor query =
                    new QueryConstructor(
                        "/search?IO_PRIO=0&json-type=dollar");
                query.append("get", get);
                query.append("prefix", context.user().prefix().toString());
                query.append("service", context.user().service());
                query.append("text", "user_id:" + context.userGuid());
                query.append("length", 1);
                if (context.debug()) {
                    context.session().logger().info(
                        "User data request: " + query);
                }

                context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    context.failoverDelay(),
                    true,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private static class ExtractUserDataCallback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;

        ExtractUserDataCallback(
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            super(context.callback());
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList result = response.get("hitsArray").asList();
                if (context.debug()) {
                    context.session().logger().info(
                        "User data lucene response: "
                            + JsonType.HUMAN_READABLE.toString(result));
                }
                Map<String, Long> clearedChats = new LinkedHashMap<>();
                Set<String> blacklistedUsers = new HashSet<>();
                if (result.size() > 0) {
                    JsonObject doc = result.get(0);
                    if (doc != null) {
                        clearedChats = doc.asMap().get(
                            UserFields.CLEARED_CHATS.stored(),
                            new LinkedHashMap<>(),
                            UserClearedChats.PARSER);
                        blacklistedUsers = doc.asMap().get(
                            UserFields.BLACKLISTED_USERS.stored(),
                            new HashSet<>(),
                            UsersSuggestRule.SETS_PARSER);
                    }
                }

                FutureCallback<Set<String>> nextCallback =
                    new PvpSuggestCallback(
                        context,
                        requestedLength,
                        hits,
                        clearedChats,
                        blacklistedUsers);

                nextCallback = new ResolveContactsForUsersCallback(
                    nextCallback,
                    context,
                    requestedLength,
                    hits);

                context.proxy().filterUserResources(
                    context,
                    context.userGuid(),
                    hits,
                    context.usersFilter(),
                    nextCallback);
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class ResolveContactsForUsersCallback
        implements FutureCallback<Set<String>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;
        private final FutureCallback<Set<String>> nextCallback;

        ResolveContactsForUsersCallback(
            final FutureCallback<Set<String>> nextCallback,
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            this.nextCallback = nextCallback;
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        @Override
        public void completed(final Set<String> result) {
            if (result.isEmpty()) {
                nextCallback.completed(result);
                context.session().logger().info(
                    "Empty results, skipping contacts resolving");
                return;
            }
            try {
                QueryConstructor query =
                    new QueryConstructor(
                        "/search?IO_PRIO=0&json-type=dollar"
                            + "&sync-searcher=false"
                            + "&skip-nulls");
                User user = context.contactsContext().user();
                query.append("get", "contact_id,contact_name");
                query.append("prefix", user.prefix().toString());
                query.append("service", user.service());
                query.append(
                    "text",
                    StringUtils.join(
                        result,
                        ' ',
                        "contact_id_p:(",
                        ")"));
                query.append("length", requestedLength);
                context.session().logger().info(
                    "Resolving contacts: " + result);

                context.proxy().sequentialRequest(
                    context.session(),
                    context.contactsContext(),
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    context.contactsContext().failoverDelay(),
                    context.contactsContext().localityShuffle(),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new ContactsCallback(
                        nextCallback,
                        context,
                        requestedLength,
                        hits,
                        result));
            } catch (HttpException e) {
                failed(e);
            }
        }

        @Override
        public void cancelled() {
            nextCallback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            nextCallback.failed(e);
        }
    }

    private static class ContactsCallback
        extends AbstractFilterFutureCallback<JsonObject, Set<String>>
    {
        private final UsersSuggestRequestContext context;
        private final JsonList usersHits;
        private final Set<String> userSet;

        ContactsCallback(
            final FutureCallback<Set<String>> nextCallback,
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList usersHits,
            final Set<String> userSet)
        {
            super(nextCallback);
            this.context = context;
            this.usersHits = usersHits;
            this.userSet = userSet;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                context.session().logger().info(
                    "ResolveContactsCallback Lucene response: "
                        + JsonType.HUMAN_READABLE.toString(hits));
                Map<String, String> contacts = null;
                if (hits.size() == 0) {
                    contacts = Collections.emptyMap();
                } else {
                    for (JsonObject jo: hits) {
                        JsonMap hit = jo.asMap();
                        String id = hit.getOrNull(CONTACT_ID);
                        String name = hit.getOrNull(CONTACT_NAME);
                        if (id != null && name != null) {
                            if (contacts == null) {
                                contacts = new HashMap<>();
                            }
                            contacts.put(id, name);
                            context.session().logger().info(
                                "Contacts.callback.put(" + id
                                    + ", " + name);
                        }
                    }
                    if (contacts == null) {
                        contacts = Collections.emptyMap();
                    }
                }
                if (!contacts.isEmpty()) {
                    for (final JsonObject hit: usersHits) {
                        JsonMap map = hit.asMap();
                        String id = map.getOrNull(RESOURCE_ID);
                        context.session().logger().info(
                            "Contacts.checking: " + id);
                        if (id != null && userSet.contains(id)) {
                            String name = contacts.get(id);
                            if (name != null) {
                                map.put(CONTACT_NAME, new JsonString(name));
                            }
                        }
                    }
                }
                callback.completed(userSet);
            } catch (JsonException e) {
                failed(e);
            }
        }
    }

    private static class PvpSuggestCallback
        extends AbstractFilterFutureCallback<Set<String>, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;
        private final Map<String, Long> clearedChats;
        private final Set<String> blacklistedUsers;

        public PvpSuggestCallback(
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits,
            final Map<String, Long> clearedChats,
            final Set<String> blacklistedUsers)
        {
            super(context.callback());
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
            this.clearedChats = clearedChats;
            this.blacklistedUsers = blacklistedUsers;
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                boolean leave = false;
                if (context.getFields().contains(field)) {
                    leave = true;
                }
                if (context.contactGetFields().contains(field)) {
                    leave = true;
                }
                if (!leave) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                final String jsonString = doc.get(UserFields.DATA.stored()).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
//                    if (contactsSearch) {
                    final JsonObject contactName =
                        doc.get(CONTACT_NAME);
                    if (contactName != JsonNull.INSTANCE) {
                        JsonMap userData = obj.asMap();
                        userData.put(DISPLAY_NAME, contactName);
                        userData.put(PASSPORT_DISPLAY_NAME, contactName);
                    }
//                    }
                    doc.put(UserFields.DATA.stored(), obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new AbstractMap.SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        @Override
        public void completed(final Set<String> resources) {
            try {
                context.session().logger().info(
                    "Request test: " + context.requestText().text()
                        + ", pvp: " + context.pvp()
                        + ", contactsSearch: " + context.contactsSearch()
                        + ", filteredResources: " + resources
                        + ", blacklistedUsers: " + blacklistedUsers);
                String request = context.requestText().text();
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));

                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    String chatId = doc.get(ChatFields.ID.stored()).asStringOrNull();
                    if (resourceId != null) {
                        if (resources.contains(resourceId)) {
                            if (blacklistedUsers.contains(resourceId)) {
                                continue;
                            }
                            long lastMsgTs =
                                doc.getLong(
                                    ChatFields.LAST_MESSAGE_TIMESTAMP.stored(),
                                    0L) / MICROS_PER_SEC;
                            if (chatId != null) {
                                Long ts = clearedChats.getOrDefault(chatId, -1L);
                                if (ts >= lastMsgTs) {
                                    lastMsgTs = 0L;
                                }
                            }
                            double score = 50.0 + doc.getDouble(SCORE, 0.0);
                            List<Map.Entry<String, String>> searchTexts =
                                searchTexts(doc);
                            jsonReformat(doc, jsonParser, consumer);
                            filterGetFields(doc);
                            UserSuggestItem item =
                                new UserSuggestItem(
                                    resourceId,
                                    context.resultsSuggestType(),
                                    request,
                                    searchTexts,
                                    score,
                                    doc,
                                    lastMsgTs);
                            items.add(item);
                        }
                    }
                }

                callback.completed(items);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}

