package ru.yandex.search.messenger.proxy.suggest.rules;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.SearchPrivacy;
import ru.yandex.search.messenger.UserFields;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.ChatsFilter;
import ru.yandex.search.messenger.proxy.suggest.MetaHandlingUsersFilter;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.UserSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.UsersFilter;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestUsersFilterProvider;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;

@SuppressWarnings("StringSplitter")
public class UsersSuggestRule
    <T extends
        RequestProvider
        & SuggestRequestContextProvider
        & SuggestUsersFilterProvider
        & SuggestChatsFilterProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final int TEAM_ORG_ID = 34;
    private static final Pattern EMAIL =
        Pattern.compile("([A-Za-z0-9._%+-]+)@([A-Za-z0-9.-]+\\.[A-Za-z]{2,4})");
    private static final long MICROS_PER_SEC = 1000000;
    private static final int MAX_USERS_PER_CHAT_LIMIT = 400;
    private static final float KEYWORD_BOOST = 5f;
    private static final String SCORE = "#score";
    private static final String B_AND_B = ") AND (";
    private static final String AND = " AND ";
    private static final String OR_B = " OR (";
    private static final String OR = " OR ";
    private static final String DP = "dp";
    private static final String POSTFILTER = "postfilter";
    private static final String USER_DATA = "user_data";
    private static final String NAME_TOKENIZED_P =
        "user_display_name_tokenized_p";
    private static final String NICKNAME_TOKENIZED_P =
        "user_nickname_tokenized_p";
    private static final String POSITION_TOKENIZED_P =
        "user_position_tokenized_p";
    private static final String WEBSITE_TOKENIZED_P =
        "user_website_tokenized_p";
    //
    private static final String NAME = "user_display_name";
    private static final String NICKNAME = "user_nickname";
    private static final String POSITION = "user_position";
    private static final String WEBSITE = "user_website";
    private static final String CONTACT_NAME = "contact_name";

    private static final String CONTACT_ID = "contact_id";
    //
    private static final String NAME_P = "user_display_name_p";
    private static final String NICKNAME_P = "user_nickname_p";
    private static final String POSITION_P = "user_position_p";
    private static final String WEBSITE_P = "user_website_p";

    private static final String DISPLAY_NAME = "display_name";
    private static final String PASSPORT_DISPLAY_NAME = "passport_display_name";

    private static final String[] SEARCH_FIELDS_TOKENIZED1 = {
        NAME_TOKENIZED_P
    };
    private static final String[] SEARCH_FIELDS_TOKENIZED2 = {
        NICKNAME_TOKENIZED_P,
        NAME_TOKENIZED_P
    };
    private static final String[] SEARCH_FIELDS_TOKENIZED3 = {
        WEBSITE_TOKENIZED_P,
        POSITION_TOKENIZED_P,
        NICKNAME_TOKENIZED_P,
        NAME_TOKENIZED_P
    };

    private static final String[] SEARCH_FIELDS_KEYWORD1 = {
        NAME_P
    };
    private static final String[] SEARCH_FIELDS_KEYWORD2 = {
        NICKNAME_P,
        NAME_P
    };
    private static final String[] SEARCH_FIELDS_KEYWORD3 = {
        WEBSITE_P,
        POSITION_P,
        NICKNAME_P,
        NAME_P
    };

    private static final String[] MATCH_FIELDS = {
        WEBSITE,
        POSITION,
        CONTACT_NAME,
        NICKNAME,
        NAME
    };
    private static final String[] LOGIN_FIELDS = {
        NAME_TOKENIZED_P,
        NICKNAME_TOKENIZED_P
    };
    private static final String[] LOGIN_FIELDS_KEYWORD = {
        NAME_P,
        NICKNAME_P
    };
//    private static final BoostByOrderFieldsSwitchingTermsSupplierFactory
//        NAME_FIELD =
//            new BoostByOrderFieldsSwitchingTermsSupplierFactory(
//                1f,
//                2f,
//                Arrays.asList(SEARCH_FIELDS_TOKENIZED),
//                Arrays.asList(SEARCH_FIELDS_KEYWORD));
//    private static final FieldsTermsSupplierFactory FIELDS =
//        new FieldsTermsSupplierFactory("folder_tokenized", NAME);
    private static final List<String> SEARCH_FIELDS_KEYWORD_LIST1 =
        Arrays.asList(SEARCH_FIELDS_KEYWORD1);
    private static final List<String> SEARCH_FIELDS_KEYWORD_LIST2 =
        Arrays.asList(SEARCH_FIELDS_KEYWORD2);
    private static final List<String> SEARCH_FIELDS_KEYWORD_LIST3 =
        Arrays.asList(SEARCH_FIELDS_KEYWORD3);
    private static final List<String> MENTION_FIELDS =
        Arrays.asList(LOGIN_FIELDS);
    private static final List<String> MENTION_FIELDS_KEYWORD_LIST =
        Arrays.asList(LOGIN_FIELDS_KEYWORD);
    private static final int MIN_LENGTH = 10;
    private static final String RESOURCE_ID = "user_id";

    public static final CollectionParser<String, Set<String>, Exception> SETS_PARSER =
        new CollectionParser<>(String::trim, HashSet::new, '\n');

    private final SynonymsFieldsTermsSupplierFactory nameField1;
    private final SynonymsFieldsTermsSupplierFactory nameField2;
    private final SynonymsFieldsTermsSupplierFactory nameField3;
    private final SynonymsFieldsTermsSupplierFactory mentionField;
    private final long failoverDelay;
    private final boolean localityShuffle;
    private final SuggestType resultsSuggestType;
    private final boolean pvp;
    private final boolean contactsSearch;

    //CSOFF: ParameterNumber
    public UsersSuggestRule(
        final Moxy moxy,
        final SuggestType resultsSuggestType,
        final boolean pvp,
        final boolean contactsSearch)
    {
        this.resultsSuggestType = resultsSuggestType;
        this.pvp = pvp;
        this.contactsSearch = contactsSearch;
        nameField1 =
            new SynonymsFieldsTermsSupplierFactory(
                1f,
                moxy.synonyms(),
                Arrays.asList(SEARCH_FIELDS_TOKENIZED1));
        nameField2 =
            new SynonymsFieldsTermsSupplierFactory(
                1f,
                moxy.synonyms(),
                Arrays.asList(SEARCH_FIELDS_TOKENIZED2));
        nameField3 =
            new SynonymsFieldsTermsSupplierFactory(
                1f,
                moxy.synonyms(),
                Arrays.asList(SEARCH_FIELDS_TOKENIZED3));
        mentionField =
            new SynonymsFieldsTermsSupplierFactory(
                1f,
                moxy.synonyms(),
                MENTION_FIELDS);
        failoverDelay = moxy.config().usersSuggestFailoverDelay();
        localityShuffle = moxy.config().usersSuggestLocalityShuffle();
    }
    //CSON: ParameterNumber

    protected static List<String> keywordFields(final int requestSize) {
        switch (requestSize) {
            case 1:
                return SEARCH_FIELDS_KEYWORD_LIST1;
            case 2:
                return SEARCH_FIELDS_KEYWORD_LIST2;
            default:
                return SEARCH_FIELDS_KEYWORD_LIST3;
        }
    }

    private SynonymsFieldsTermsSupplierFactory searchFields(
        final int requestSize)
    {
        switch (requestSize) {
            case 1:
                return nameField1;
            case 2:
                return nameField2;
            default:
                return nameField3;
        }
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> acallback)
        throws HttpException
    {
        FutureCallback<? super List<SuggestItem>> callback = acallback;
        boolean global = resultsSuggestType == SuggestType.USERS_GLOBAL;
        if (global) {
            final PrefixedLogger logger = input.suggestRequestContext().session().logger();
            callback = new FutureCallback<>() {
                @Override
                public void completed(final List<SuggestItem> suggestItems) {
                    logger.info("Global completed " + suggestItems.size() + " " + acallback);
                    acallback.completed(suggestItems);
                }

                @Override
                public void failed(final Exception e) {
                    logger.log(Level.WARNING, "Global failed ", e);
                    acallback.failed(e);
                }

                @Override
                public void cancelled() {
                    logger.info("Global cancelled");
                    acallback.cancelled();
                }
            };
        }
        SuggestRequestContext context = input.suggestRequestContext();
        if (input.usersFilter().empty()) {
            context.logger().info("Empty users list");
            callback.completed(Collections.emptyList());
            return;
        }
        String request = input.request();
        if (!contactsSearch
            && context.hadRequest(resultsSuggestType, request))
        {
            context.logger().info("Request: " + request
                + "has already been executed. Skipping");
            callback.completed(Collections.emptyList());
            return;
        }


        execute(
            Math.max(MIN_LENGTH, context.length() << 1),
            new UsersSuggestRequestContext(
                context,
                request,
                SearchRequestText.parseSuggest(input.request(), false),
                input.usersFilter(),
                input.chatsFilter(),
                pvp,
                failoverDelay,
                localityShuffle,
                resultsSuggestType,
                callback,
                contactsSearch,
                global));
    }

    public void execute(
        final int requestedLength,
        final UsersSuggestRequestContext context)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder();
        SearchRequestText request = context.request;
        int requestSize = request.text().length();
        if (request.isEmpty() && requestSize > 0) {
            context.logger().info("No human words in  Request: " + request
                + " .Skipping ");
            context.callback.completed(Collections.emptyList());
            return;
        }
        if (context.queryFilter != null) {
            sb.append('(');
            sb.append(context.queryFilter);
            sb.append(')');
        }
        if (!context.contactsSearch) {
            boolean closeBrace = false;
            if (request.hasMentions() || request.hasWords()) {
                if (sb.length() > 0) {
                    sb.append(AND);
                }
                sb.append('(');
                closeBrace = true;
            }
            boolean mboxSearch = false;
            if (context.mboxSearch != null) {
                mboxSearch = true;
                sb.append(NICKNAME_TOKENIZED_P);
                sb.append(":(");
                sb.append(context.mboxSearch);
                sb.append(") OR ");
                sb.append(NICKNAME_P);
                sb.append(":(");
                sb.append(context.mboxSearch);
                sb.append(')');
            }
            if (request.hasMentions() && !mboxSearch) {
                sb.append('(');
                request.mentionsQuery(
                    sb,
                    mentionField,
                    B_AND_B);
                sb.append(')');
                if (request.singleMentionWord()) {
                    System.err.println("Single mention word");
                    sb.append(OR_B);
                    request.mentionsQuery(
                        sb,
                        MENTION_FIELDS_KEYWORD_LIST,
                        B_AND_B,
                        KEYWORD_BOOST);
                    sb.append(')');
                }
            }
            if (request.hasWords() && !mboxSearch) {
                if (request.hasMentions()) {
                    sb.append(OR);
                }
                sb.append('(');
                request.fieldsQuery(
                    sb,
                    searchFields(requestSize),
                     B_AND_B);
                sb.append(')');
                if (request.singleWord()) {
                    sb.append(OR_B);
                    request.fieldsQuery(
                        sb,
                        keywordFields(requestSize),
                        B_AND_B,
                        KEYWORD_BOOST);
                    sb.append(')');
                }
            }
            if (closeBrace) {
                sb.append(')');
            }
            if (sb.length() == 0) {
                context.logger().info("Empty token list for Request: " + request
                    + " .Skipping");
                context.callback.completed(Collections.emptyList());
                return;
            }
        }
        if (context.contactsSearch
            && context.usersFilter.users().size() == 0
            && context.chatsFilter.chats().size() == 0)
        {
            context.logger().info(
                "Contacts search and filters size == 0 .Skipping");
            context.callback.completed(Collections.emptyList());
            return;
        }
        if (context.usersFilter.users().size() > 0
            && context.usersFilter.users().size() < MAX_USERS_PER_CHAT_LIMIT)
        {
            String sep = "";
            if (sb.length() == 0) {
                sb.append("user_id:(");
            } else {
                sb.append(" AND user_id:(");
            }
            for (String user: context.usersFilter.users()) {
                sb.append(sep);
                sb.append(user);
                sep = " ";
            }
            sb.append(')');
        }
        if (context.chatsFilter.chats().size() == 1) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            sb.append("user_chats:");
            sb.append(context.chatsFilter.chats().keySet().iterator().next());
        }
        sb.append(" AND NOT ");
        sb.append(UserFields.SEARCH_PRIVACY.global());
        sb.append(':');
        sb.append(SearchPrivacy.NOBODY.value());

//        request.negationsQuery(sb, NAME_FIELD);
        LinkedHashSet<String> get = new LinkedHashSet<>();
        get.add(RESOURCE_ID);
        get.add(NAME);
        get.add(NICKNAME);
        get.add(POSITION);
        get.add(WEBSITE);
        get.add(UserFields.SEARCH_PRIVACY.stored());
        if (!context.contactsSearch) {
            get.add(SCORE);
        }
        get.addAll(context.getFields);
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                    + "&sync-searcher=false"
                    + "&skip-nulls");
//                    + "&scorer=lucene&sort=multi(%23score,user_nickname)"
//                    + "&dp=contains(id,@+has_at)&postfilter=has_at+==+1"
//                    + "&dp=fallback(user_id+id)");
        User user = context.user();
        if (!context.contactsSearch) {
            query.append("scorer", "lucene");
            query.append(DP, "contains(id,@ has_at)");
            query.append(POSTFILTER, "has_at == 1");
            query.append(DP, "fallback(user_id id)");
            query.append(DP, "equals(user_is_robot,true is_robot)");
            query.append(DP, "const(30 down)");
            query.append(DP, "const(1 one)");
//            query.append(DP, "to_long(one one)");
            query.append(DP, "if(is_robot,down,one score_down)");
            query.append(DP, "fdiv(#score,score_down #score)");
//            query.append("get", "is_robot,score_down,one,score");
        }
        if (context.v2Org() != null && context.v2Org() != 0) {
            query.append("db", "v2org");
        }
        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        String searchText = new String(sb);
        if (context.v2Org() != null && context.v2Org() != 0) {
            query.append("text", "type_p:user AND (" + searchText + ')');
        } else {
            query.append("text", searchText);
        }

        context.logger().fine("UsersSuggestRule: request="
            + context.request()
            + ", backend_request: " + searchText);
        query.append("length", requestedLength);
        if (context.dps != null && context.dps.size() > 0) {
            for (String dp: context.dps) {
                query.append(DP, dp);
            }
        }
        if (context.postfilters != null && context.postfilters.size() > 0) {
            for (String pf: context.postfilters) {
                query.append(POSTFILTER, pf);
            }
        }
        if (context.global && requestSize < 3) {
            query.append("early-interrupt", "true");
        }
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(context, requestedLength));
    }

    private static class UsersSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final SearchRequestText request;
//        private final String type;
        private final FutureCallback<? super List<SuggestItem>> callback;
        private final User user;
        private final UsersFilter usersFilter;
        private final ChatsFilter chatsFilter;
        private final Set<String> getFields;
        private final Set<String> contactGetFields;
        private final String queryFilter;
        private final List<String> dps;
        private final List<String> postfilters;
        private final long failoverDelay;
        private final boolean localityShuffle;
        private final SuggestType resultsSuggestType;
        private final boolean contactsSearch;
        private final boolean pvp;
        private final ContactsSuggestRequestContext contactsContext;
        private final String mboxSearch;
        private final String userGuid;
        private final boolean global;
        private final String database;

        // CSOFF: ParameterNumber
        UsersSuggestRequestContext(
            final SuggestRequestContext suggestRequestContext,
            final String requestString,
            final SearchRequestText request,
            final UsersFilter usersFilter,
            final ChatsFilter chatsFilter,
            final boolean pvp,
            final long failoverDelay,
            final boolean localityShuffle,
            final SuggestType resultsSuggestType,
            final FutureCallback<? super List<SuggestItem>> callback,
            final boolean contactsSearch,
            final boolean global)
            throws BadRequestException
        {
            super(suggestRequestContext, requestString);
            this.request = request;
            this.callback = callback;
            this.usersFilter = usersFilter;
            this.chatsFilter = chatsFilter;
            this.resultsSuggestType = resultsSuggestType;
            this.contactsSearch = contactsSearch;
            this.pvp = pvp;
            this.global = global;
            String mboxSearch = null;
            long userId = 0;
            if (suggestRequestContext.v2Org() != null) {
                userId = suggestRequestContext.v2Org();
                database = "v2org";
            } else {
                if (!contactsSearch && !pvp) {
                    String organizations =
                        session().params().getString("organizations", null);
                    if (organizations != null) {
                        String[] orgs = organizations.split(",");
                        String orgId = orgs[0].trim();
                        try {
                            userId = Long.parseLong(orgId);
                        } catch (NumberFormatException e) {
                            //ignore
                            userId = 0;
                        }
                    }
                }
                database = null;
            }

            // team
            if (userId == TEAM_ORG_ID) {
                Matcher m = EMAIL.matcher(requestString);
                if (m.matches()) {
                    String mbox = m.group(1);
                    String host = m.group(2);
                    session().logger().info(
                        "Dropping email's domain part from: "
                            + requestString
                            + ", mbox: " + mbox
                            + ", host: " + host);
                    if (host.equalsIgnoreCase("yandex-team.ru")
                            || host.equalsIgnoreCase("yandex-team.com")
                            || host.equalsIgnoreCase("auto.ru")
                            || host.equalsIgnoreCase("yamoney.ru"))
                    {
                        mboxSearch = mbox;
                    }
                }
            }
            if (global && session().params().getBoolean("team", false)) {
                userId = TEAM_ORG_ID;
            }

            this.mboxSearch = mboxSearch;
            if (suggestRequestContext.v2Org() != null && suggestRequestContext.v2Org() != 0) {
                user = new User(
                    suggestRequestContext.proxy().config().v2UsersService(),
                    new LongPrefix(suggestRequestContext.v2Org()));
                session().logger().info("v2user " + user);
            } else {
                user = new User(
                    suggestRequestContext.proxy().config().usersService(),
                    new LongPrefix(userId));
            }

            getFields =
                session().params().get(
                    "user_get",
                    new LinkedHashSet<>(),
                    new CollectionParser<>(
                        String::trim,
                        LinkedHashSet::new,
                        ','));
            if (contactsSearch) {
                contactGetFields =
                    session().params().get(
                        "contact_get",
                        new LinkedHashSet<>(),
                        new CollectionParser<>(
                            String::trim,
                            LinkedHashSet::new,
                            ','));
                contactGetFields.add(CONTACT_NAME);
            } else {
                contactGetFields = Collections.emptySet();
            }
            if (pvp) {
                String userFilter = session().params().getString("pvp_user_filter", null);
                if (v2Org() != null && v2Org() != 0 && userFilter != null) {
                    userFilter = userFilter.replaceFirst("user_org_id:0", "user_org_id:" + v2Org());
                }

                queryFilter = userFilter;

                dps = session().params().getAll("pvp_user_dp");
                postfilters = session().params().getAll("pvp_user_postfilter");
            } else if (contactsSearch) {
                queryFilter =
                    session().params().getString(
                        "contacts_user_filter",
                        "user_org_id:0 AND user_is_dismissed:false"
                            + " AND user_is_robot:false");
                dps = session().params().getAll("contacts_user_dp");
                postfilters =
                    session().params().getAll("contacts_user_postfilter");
            } else {
                String userFilter = session().params().getString("user_filter", null);
                if (v2Org() != null && v2Org() != 0 && userFilter != null) {
                    userFilter = userFilter.replaceFirst("user_org_id:0", "user_org_id:" + v2Org());
                }
                queryFilter = userFilter;
                dps = session().params().getAll("user_dp");
                postfilters = session().params().getAll("user_postfilter");
            }
            this.failoverDelay = session().params().getLongDuration(
                "user_failover_delay",
                failoverDelay);
            this.localityShuffle = session().params().getBoolean(
                "user_locality_shuffle",
                localityShuffle);
            userGuid = session().params().getString(
                ResolveContactsUsersRule.REQUEST_USER_ID,
                null);
            if (userGuid == null) {
                contactsContext = null;
            } else {
                contactsContext =
                    new ContactsSuggestRequestContext(
                        userGuid,
                        suggestRequestContext,
                        failoverDelay,
                        localityShuffle);
            }
        }
        // CSON: ParameterNumber

        @Override
        public User user() {
            return user;
        }
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;

        Callback(
            final UsersSuggestRequestContext context,
            final int requestedLength)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                if (context.debug()) {
                    context.session().logger().info(
                        "Lucene response: "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                ExtractUserDataCallback callback =
                    new ExtractUserDataCallback(
                        context,
                        requestedLength,
                        hits);

                QueryConstructor query =
                    new QueryConstructor(
                        "/search?IO_PRIO=0&json-type=dollar");
                query.append("get", UserFields.BLACKLISTED_USERS.stored());
                query.append("prefix", context.user().prefix().toString());
                query.append("service", context.user().service());
                query.append("text", "user_id:" + context.userGuid);
                query.append("length", 1);
                if (context.database != null) {
                    query.append("db", context.database);
                }
                if (context.debug()) {
                    context.session().logger().info(
                        "User data request: " + query);
                }

                context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    context.failoverDelay,
                    true,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    callback);
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }
    }

    private class ExtractUserDataCallback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;

        ExtractUserDataCallback(
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList result = response.get("hitsArray").asList();
                if (context.debug()) {
                    context.session().logger().info(
                        "User data lucene response: "
                            + JsonType.HUMAN_READABLE.toString(result));
                }
                Set<String> blacklistedUsers = new HashSet<>();
                if (result.size() > 0) {
                    JsonObject doc = result.get(0);
                    if (doc != null) {
                        blacklistedUsers = doc.asMap().get(
                            UserFields.BLACKLISTED_USERS.stored(),
                            new HashSet<>(),
                            SETS_PARSER);
                    }
                }

                FutureCallback<Set<String>> nextCallback =
                    new FilteredResourcesCallback(
                        context,
                        requestedLength,
                        hits,
                        blacklistedUsers);
                if (!context.contactsSearch
                    && context.contactsContext != null)
                {
                    nextCallback = new ResolveContactsForUsersCallback(
                        nextCallback,
                        context,
                        requestedLength,
                        hits);
                }
                context.proxy().filterUserResources(
                    context,
                    context.userGuid,
                    hits,
                    context.usersFilter,
                    nextCallback);
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class ResolveContactsForUsersCallback
        implements FutureCallback<Set<String>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;
        private final FutureCallback<Set<String>> nextCallback;

        ResolveContactsForUsersCallback(
            final FutureCallback<Set<String>> nextCallback,
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits)
        {
            this.nextCallback = nextCallback;
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        @Override
        public void completed(final Set<String> result) {
            if (result.isEmpty()) {
                nextCallback.completed(result);
                    context.session().logger().info(
                        "Empty results, skipping contacts resolving");
                return;
            }
            try {
                QueryConstructor query =
                    new QueryConstructor(
                        "/search?IO_PRIO=0&json-type=dollar"
                            + "&sync-searcher=false"
                            + "&skip-nulls");
                User user = context.contactsContext.user();
                query.append("get", "contact_id,contact_name");
                query.append("prefix", user.prefix().toString());
                query.append("service", user.service());
                query.append(
                    "text",
                    StringUtils.join(
                        result,
                        ' ',
                        "contact_id_p:(",
                        ")"));
                query.append("length", requestedLength);
                context.session().logger().info(
                    "Resolving contacts: " + result);

                context.proxy().sequentialRequest(
                    context.session(),
                    context.contactsContext,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    context.contactsContext.failoverDelay,
                    context.contactsContext.localityShuffle,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new ContactsCallback(
                        nextCallback,
                        context,
                        requestedLength,
                        hits,
                        result));
            } catch (HttpException e) {
                failed(e);
            }
        }

        @Override
        public void cancelled() {
            nextCallback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            nextCallback.failed(e);
        }
    }

    private class FilteredResourcesCallback
        extends AbstractFilterFutureCallback<Set<String>, List<SuggestItem>>
    {
        private final UsersSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;
        private final Set<String> blacklistedUsers;

        FilteredResourcesCallback(
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList hits,
            final Set<String> blacklistedUsers)
        {
            super(context.callback);
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
            this.blacklistedUsers = blacklistedUsers;
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                boolean leave = false;
                if (context.getFields.contains(field)) {
                    leave = true;
                }
                if (context.contactGetFields.contains(field)) {
                    leave = true;
                }
                if (!leave) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                final String jsonString = doc.get(USER_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
//                    if (contactsSearch) {
                    final JsonObject contactName =
                        doc.get(CONTACT_NAME);
                    if (contactName != JsonNull.INSTANCE) {
                        JsonMap userData = obj.asMap();
                        userData.put(DISPLAY_NAME, contactName);
                        userData.put(PASSPORT_DISPLAY_NAME, contactName);
                    }
//                    }
                    doc.put(USER_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        @Override
        public void completed(final Set<String> resources) {
            try {
                context.session().logger().info(
                    "Request test: " + context.request.text()
                    + ", pvp: " + context.pvp
                    + ", contactsSearch: " + context.contactsSearch
                    + ", filteredResources: " + resources
                    + ", blacklistedUsers: " + blacklistedUsers);
                String request = context.request.text();
                int count = 0;
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));

                MetaHandlingUsersFilter metaFeed = null;
                if (context.usersFilter instanceof MetaHandlingUsersFilter) {
                    metaFeed = (MetaHandlingUsersFilter) context.usersFilter;
                }
                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    if (resourceId != null) {
                        ++count;
                        if (blacklistedUsers.contains(resourceId)) {
                            continue;
                        }
                        if (resources.contains(resourceId)) {
                            if (metaFeed != null) {
                                metaFeed.feedMeta(resourceId, doc);
                            }
//                            String text = selectText(doc);
                            long lastMsgTs =
                                doc.getLong(
                                    "chat_last_message_timestamp",
                                    0L) / MICROS_PER_SEC;
                            double score = 0;
                            if (contactsSearch) {
                                score = 40.0;
                            }
                            score += doc.getDouble(SCORE, 0.0);
                            List<Map.Entry<String, String>> searchTexts =
                                searchTexts(doc);
                            jsonReformat(doc, jsonParser, consumer);
                            filterGetFields(doc);
                            UserSuggestItem item =
                                new UserSuggestItem(
                                    resourceId,
                                    context.resultsSuggestType,
                                    request,
                                    searchTexts,
                                    score,
                                    doc,
                                    lastMsgTs);
                            items.add(item);
                        }
                    }
                }
                if (items.size() >= context.length()
                    || count < requestedLength)
                {
                    callback.completed(items);
                } else {
                    execute(requestedLength << 1, context);
                }
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class ContactsSuggestRequestContext
        extends BasicSuggestRequestContext
    {
        private final User user;
        private final long failoverDelay;
        private final boolean localityShuffle;

        // CSOFF: ParameterNumber
        ContactsSuggestRequestContext(
            final String userId,
            final SuggestRequestContext suggestRequestContext,
            final long failoverDelay,
            final boolean localityShuffle)
            throws BadRequestException
        {
            super(suggestRequestContext, "");
            user = new User(
                suggestRequestContext.proxy().config().messagesService(),
                new StringPrefix(userId));
            this.failoverDelay = session().params().getLongDuration(
                "contact_failover_delay",
                failoverDelay);
            this.localityShuffle = session().params().getBoolean(
                "contact_locality_shuffle",
                localityShuffle);
        }
        // CSON: ParameterNumber

        @Override
        public User user() {
            return user;
        }
    }

    private static class ContactsCallback
        extends AbstractFilterFutureCallback<JsonObject, Set<String>>
    {
        private final UsersSuggestRequestContext context;
        private final JsonList usersHits;
        private final Set<String> userSet;

        ContactsCallback(
            final FutureCallback<Set<String>> nextCallback,
            final UsersSuggestRequestContext context,
            final int requestedLength,
            final JsonList usersHits,
            final Set<String> userSet)
        {
            super(nextCallback);
            this.context = context;
            this.usersHits = usersHits;
            this.userSet = userSet;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.get("hitsArray").asList();
                context.session().logger().info(
                    "ResolveContactsCallback Lucene response: "
                        + JsonType.HUMAN_READABLE.toString(hits));
                Map<String, String> contacts = null;
                if (hits.size() == 0) {
                    contacts = Collections.emptyMap();
                } else {
                    for (JsonObject jo: hits) {
                        JsonMap hit = jo.asMap();
                        String id = hit.getOrNull(CONTACT_ID);
                        String name = hit.getOrNull(CONTACT_NAME);
                        if (id != null && name != null) {
                            if (contacts == null) {
                                contacts = new HashMap<>();
                            }
                            contacts.put(id, name);
                            context.session().logger().info(
                                "Contacts.callback.put(" + id
                                    + ", " + name);
                        }
                    }
                    if (contacts == null) {
                        contacts = Collections.emptyMap();
                    }
                }
                if (!contacts.isEmpty()) {
                    for (final JsonObject hit: usersHits) {
                        JsonMap map = hit.asMap();
                        String id = map.getOrNull(RESOURCE_ID);
                        context.session().logger().info(
                            "Contacts.checking: " + id);
                        if (id != null && userSet.contains(id)) {
                            String name = contacts.get(id);
                            if (name != null) {
                                map.put(CONTACT_NAME, new JsonString(name));
                            }
                        }
                    }
                }
                callback.completed(userSet);
            } catch (JsonException e) {
                failed(e);
            }
        }
    }
}
