package ru.yandex.search.messenger.proxy.suggest.rules.chats;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.search.messenger.ChatUtils;
import ru.yandex.search.messenger.UserChats;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.request.util.BoostByOrderFieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.RequestInfo;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestInfosProvider;
import ru.yandex.util.string.StringUtils;

public class ChatsSuggestRule2
    <T extends
        RequestInfosProvider
        & SuggestRequestContextProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final long MAX_CACHE_SIZE = 1024 * 1024 * 1024;
    private static final long CHATS_CACHE_TIME = 15000;
    private static final double STARTING_BOOST = 32;
    private static final String B_AND_B = ") AND (";
    private static final String SCORE = "score";
    private static final String LUCENE_SCORE = "#score";
    private static final String CHAT_MEMBER_COUNT = "chat_member_count";
    private static final String CHAT_DATA = "chat_data";
    private static final String DESCRIPTION_TOKENIZED =
        "chat_description_tokenized";
    //
    private static final String NAME = "chat_name";
    private static final String NAME_TOKENIZED = "chat_name_tokenized";
    private static final String DESCRIPTION = "chat_description";
    private static final String DESCRIPTION_TOKENIZED_P =
        "chat_description_tokenized_p";
    //
    private static final String NAME_P = "chat_name_p";
    private static final String NAME_TOKENIZED_P = "chat_name_tokenized_p";
    private static final String DESCRIPTION_P = "chat_description_p";

//    private static final String[] TOKENIZED_SEARCH_FIELDS = {
//        DESCRIPTION_TOKENIZED,
//        NAME_TOKENIZED
//    };
    private static final String[] CHANNELS_TOKENIZED_SEARCH_FIELDS = {
        NAME_TOKENIZED
    };

//    private static final String[] KEYWORD_SEARCH_FIELDS = {
//        DESCRIPTION,
//        NAME
//    };

    private static final String[] CHANNELS_KEYWORD_SEARCH_FIELDS = {
        NAME
    };

    private static final String[] MATCH_FIELDS = {
        DESCRIPTION,
        NAME
    };
    private static final String[] CHANNELS_TOKENIZED_SEARCH_FIELDS_P = {
        NAME_TOKENIZED_P
    };

//    private static final String[] KEYWORD_SEARCH_FIELDS_P = {
//        DESCRIPTION_P,
//        NAME_P
//    };

    private static final String[] CHANNELS_KEYWORD_SEARCH_FIELDS_P = {
        NAME_P
    };

    private static final BoostByOrderFieldsTermsSupplierFactory
        DESCRIPTIONLESS_KEYWORD_FIELDS =
        new BoostByOrderFieldsTermsSupplierFactory(
            2f,
            2f,
            Arrays.asList(CHANNELS_KEYWORD_SEARCH_FIELDS));

    private static final BoostByOrderFieldsTermsSupplierFactory
        DESCRIPTIONLESS_TOKENIZED_FIELDS =
        new BoostByOrderFieldsTermsSupplierFactory(
            2f,
            1f,
            Arrays.asList(CHANNELS_TOKENIZED_SEARCH_FIELDS));
    private static final BoostByOrderFieldsTermsSupplierFactory
        DESCRIPTIONLESS_KEYWORD_FIELDS_P =
        new BoostByOrderFieldsTermsSupplierFactory(
            2f,
            2f,
            Arrays.asList(CHANNELS_KEYWORD_SEARCH_FIELDS_P));

    private static final BoostByOrderFieldsTermsSupplierFactory
        DESCRIPTIONLESS_TOKENIZED_FIELDS_P =
        new BoostByOrderFieldsTermsSupplierFactory(
            2f,
            1f,
            Arrays.asList(CHANNELS_TOKENIZED_SEARCH_FIELDS_P));
//    private static final BoostByOrderFieldsTermsSupplierFactory
//        TOKENIZED_FIELDS =
//            new BoostByOrderFieldsTermsSupplierFactory(
//                2f,
//                1f,
//                Arrays.asList(TOKENIZED_SEARCH_FIELDS));
//    private static final BoostByOrderFieldsTermsSupplierFactory
//        KEYWORD_FIELDS =
//            new BoostByOrderFieldsTermsSupplierFactory(
//                2f,
//                2f,
//                Arrays.asList(KEYWORD_SEARCH_FIELDS));


    private static final int MIN_LENGTH = 10;
    private static final String CHAT_ID = "chat_id";
    private static final String RESOURCE_ID = CHAT_ID;

    private static final String[] SAVED_MESSAGES = {
        "сохраненные сообщения",
        "cохраненки",
        "избранное",
        "сообщения сохраненные",
        "favorites",
        "favorite",
        "saved messages",
        "messages saved",
        "messages"
    };
    private static final double SAVED_MESSAGES_SCORE = 100000.0;

    private final long failoverDelay;
    private final boolean localityShuffle;
    private final SuggestType suggestType;
    private final User user;
    private final Cache<String, Set<String>> channelsCache;
    private final BoostByOrderFieldsTermsSupplierFactory keywordsFields;
    private final BoostByOrderFieldsTermsSupplierFactory tokenizedFields;
    private final BoostByOrderFieldsTermsSupplierFactory v2OrgTokenizedFields;
    private final BoostByOrderFieldsTermsSupplierFactory v2OrgKeywordsFields;

    public ChatsSuggestRule2(final Moxy moxy, final SuggestType suggestType) {
        this.suggestType = suggestType;
        failoverDelay = moxy.config().chatsSuggestFailoverDelay();
        localityShuffle = moxy.config().chatsSuggestLocalityShuffle();
        user = new User(
            moxy.config().chatsService(),
            UserChats.prefix());
        channelsCache = CacheBuilder.newBuilder()
            .maximumSize(MAX_CACHE_SIZE)
            .expireAfterWrite(
                CHATS_CACHE_TIME,
                TimeUnit.MILLISECONDS)
            .build();

        keywordsFields = DESCRIPTIONLESS_KEYWORD_FIELDS;
        tokenizedFields = DESCRIPTIONLESS_TOKENIZED_FIELDS;
        v2OrgKeywordsFields = DESCRIPTIONLESS_KEYWORD_FIELDS_P;
        v2OrgTokenizedFields = DESCRIPTIONLESS_TOKENIZED_FIELDS_P;
    }

    protected void fetchUserChats(
        final ChatsSuggestRequestContext context,
        final int requestedLength,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        Set<String> chats = channelsCache.getIfPresent(context.requestUserId());
        if (chats != null) {
            onChannelsExtracted(context, requestedLength, chats, callback);
            return;
        }

        ProxySession session = context.session();
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                    + "&sync-searcher=false"
                    + "&skip-nulls");
        String prefix;
        String service;
        if (context.v2Org() != null && context.v2Org() !=0) {
            prefix = Long.toString(context.v2Org());
            service = context.suggestRequestContext().proxy().config().v2ChatsService();
        } else {
            prefix = user.prefix().toString();
            service = user.service();
        }
        query.append("get", UserChats.CHATS.stored());
        query.append("prefix", prefix);
        query.append("service", service);
        query.append("text", "id:" + UserChats.id(context.requestUserId()));
        query.append("length", "1");

        context.proxy().sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            failoverDelay,
            localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new UserChatsCallback(context, requestedLength, callback));
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        List<RequestInfo> requests = input.requests();
//        if (context.hadRequest(suggestType, request)) {
//            context.logger().info("Request: " + request
//                + "has already been executed. Skipping");
//            callback.completed(Collections.emptyList());
//            return;
//        }

        ChatsSuggestRequestContext chatsContext = new ChatsSuggestRequestContext(
            context,
            requests,
            //SearchRequestText.parseSuggest(input.request(), false),
            failoverDelay,
            localityShuffle,
            callback,
            suggestType);

        SearchRequestText originalRequest = chatsContext.parsedRequests().get(0);
        if (originalRequest.isEmptyIgnoreMentions()
            && originalRequest.text().length() > 0) {
            context.logger().info("No human words in  Request: "
                + originalRequest + " .Skipping ");
            callback.completed(Collections.emptyList());
            return;
        }
        if (originalRequest.text().length() < 2) {
            context.logger().info("Skipping search for short requests: "
                + originalRequest);
            callback.completed(Collections.emptyList());
            return;
        }

        int requestedLength = Math.max(MIN_LENGTH, context.length() << 1);
        // here the trick for new scheme for member indexations
        // for channels we not store inside chat document
        // so if have filter with members check we should split logic
        if (chatsContext.v2Org() != null
                || chatsContext.channelsSearch()
            || chatsContext.requestUserId() == null
            || chatsContext.queryFilter() == null
            || !chatsContext.queryFilter().contains("members"))
        {
            execute(requestedLength, chatsContext);
        } else {
            DoubleFutureCallback<List<SuggestItem>, List<SuggestItem>> dfcb =
                new DoubleFutureCallback<>(
                    new ConcatCallback(context.session(), chatsContext.callback()));
            execute(
                requestedLength,
                chatsContext,
                chatsContext.queryFilter(),
                dfcb.first());
            fetchUserChats(chatsContext, requestedLength, dfcb.second());
        }
    }

    private String generateRequest(
        final List<SearchRequestText> requests,
        final BoostByOrderFieldsTermsSupplierFactory keywordsFields,
        final BoostByOrderFieldsTermsSupplierFactory tokenizedFields)
    {
        StringBuilder sb = null;
        double boost = STARTING_BOOST;
        for (SearchRequestText request: requests) {
            if (request.hasWords()) {
                if (sb == null) {
                    sb = new StringBuilder("(");
                } else {
                    sb.append(" OR ");
                }
                sb.append("(");
                request.fieldsQuery(
                    sb,
                    tokenizedFields,
                     B_AND_B);
//                sb.append(')');
                if (request.singleWord()) {
                    sb.append(" OR ");
                    request.fieldsQuery(
                        sb,
                        keywordsFields,
                        B_AND_B);
//                    sb.append(')');
                }
                sb.append(")^" + boost);
                boost /= 2;
            }
        }
        if (sb != null) {
            sb.append(')');
            return new String(sb);
        } else {
            return null;
        }
    }

    public void execute(
        final int requestedLength,
        final ChatsSuggestRequestContext context)
        throws HttpException
    {
        execute(requestedLength, context, context.queryFilter(), context.callback());
    }

    public void execute(
        final int requestedLength,
        final ChatsSuggestRequestContext context,
        final String quryFilter,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder("");
        SearchRequestText originalRequest = context.parsedRequests().get(0);
        if (originalRequest.isEmptyIgnoreMentions()
            && originalRequest.text().length() > 0) {
            context.logger().info("No human words in  Request: "
                + originalRequest + " .Skipping ");
            context.callback().completed(Collections.emptyList());
            return;
        }
        if (originalRequest.text().length() < 2) {
            context.logger().info("Skipping search for short requests: "
                + originalRequest);
            context.callback().completed(Collections.emptyList());
            return;
        }
        if (context.channelsSearch()) {
//            sb.append("chat_id:1/* AND chat_show_on_morda:true");
            //TODO: fix: index chat_is_channel flag
            sb.append("chat_show_on_morda:true");
        }
        if (quryFilter != null) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            sb.append('(');
            sb.append(quryFilter);
            sb.append(')');
        }
        String requestText;
        if (context.v2Org() != null) {
            requestText = generateRequest(context.parsedRequests(), v2OrgKeywordsFields, v2OrgTokenizedFields);
        } else {
            requestText = generateRequest(context.parsedRequests(), keywordsFields, tokenizedFields);
        }
        if (requestText != null) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            sb.append(requestText);
        }
        if (sb.length() == 0) {
            context.logger().info("Empty token list for Request: "
                + originalRequest + " .Skipping");
            context.callback().completed(Collections.emptyList());
            return;
        }

        if (context.namespaces() != null) {
            if (!context.namespaces().isEmpty()) {
                if (sb.length() > 0) {
                    sb.append(" AND ");
                }
                sb.append(ChatFields.NAMESPACE.global());
                sb.append(":(");
                for (String namespace: context.namespaces()) {
                    sb.append(namespace);
                    sb.append(" ");
                }

                sb.setLength(sb.length() - 1);
                sb.append(')');
            } else if (!context.namespacesGuids().isEmpty()) {
                if (sb.length() > 0) {
                    sb.append(" AND ");
                }

                sb.append("id");
                sb.append(":(");
                for (String guid: context.namespacesGuids()) {
                    sb.append("chat_");
                    sb.append(context.requestUserId());
                    sb.append('_');
                    sb.append(guid);
                    sb.append(' ');
                    sb.append("chat_");
                    sb.append(guid);
                    sb.append('_');
                    sb.append(context.requestUserId());
                    sb.append(' ');
                }
                sb.setLength(sb.length() - 1);
                sb.append(')');
            } else {
                context.logger().info("Empty namespace guids and empty namespaces");
                context.callback().completed(Collections.emptyList());
                return;
            }
        }
//        context.logger().info("Channel search " + context.channelsSearch()
//        if (!context.channelsSearch()) {
//            sb.append(" AND NOT chat_channel:true");
//        }

//        request.negationsQuery(sb, NAME_FIELD);
        LinkedHashSet<String> get = new LinkedHashSet<>();
        get.add(RESOURCE_ID);
        get.add(NAME);
        get.add(DESCRIPTION);
        get.add(SCORE);
        get.add(LUCENE_SCORE);
        get.add(ChatFields.LAST_MESSAGE_TIMESTAMP.stored());
        get.addAll(context.getFields());
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                + "&sync-searcher=false"
                + "&skip-nulls"
                + "&scorer=lucene"
                + "&dp=fallback(chat_id+id)");
        String fallback = "fallback(";
        if (context.channelsSearch()) {
            fallback += CHAT_MEMBER_COUNT + ',';
        } else {
            fallback += ChatFields.LAST_MESSAGE_TIMESTAMP.stored() + ',';
        }
        fallback += LUCENE_SCORE + ' ' + SCORE + ')';
        query.append("dp", fallback);
        query.append("sort", SCORE);
        User user = context.user();
        String queryText = new String(sb);
        query.append("get", StringUtils.join(get, ','));
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        if (context.v2Org() != null) {
            query.append("db", "v2org");
            query.append("text", "type_p:chat AND (" + queryText + ")");
        } else {
            query.append("text", queryText);
        }

        query.append("length", requestedLength);

        if (context.dps() != null && context.dps().size() > 0) {
            for (String dp: context.dps()) {
                query.append("dp", dp);
            }
        }
        if (context.postfilters() != null && context.postfilters().size() > 0) {
            for (String pf: context.postfilters()) {
                query.append("postfilter", pf);
            }
        }

        context.logger().info("Generated request: " + queryText);
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.failoverDelay(),
            context.localityShuffle(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new Callback(context, callback, requestedLength));
    }

    private class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final ChatsSuggestRequestContext context;
        private final int requestedLength;

        Callback(
            final ChatsSuggestRequestContext context,
            final FutureCallback<? super List<SuggestItem>> callback,
            final int requestedLength)
        {
            super(callback);
            this.context = context;
            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                context.proxy().filterChatResources(
                    context,
                    hits,
                    new FilteredResourcesCallback(
                        context,
                        callback,
                        requestedLength,
                        hits));
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private class FilteredResourcesCallback
        extends AbstractFilterFutureCallback<Set<String>, List<SuggestItem>>
    {
        private final ChatsSuggestRequestContext context;
        private final int requestedLength;
        private final JsonList hits;

        FilteredResourcesCallback(
            final ChatsSuggestRequestContext context,
            final FutureCallback<? super List<SuggestItem>> callback,
            final int requestedLength,
            final JsonList hits)
        {
            super(callback);
            this.context = context;
            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                if (!context.getFields().contains(field)) {
                    docIter.remove();
                }
            }
        }

        private void jsonReformat(
            final JsonMap doc,
            final JsonParser jsonParser,
            final BasicGenericConsumer<JsonObject, JsonException> consumer)
        {
            try {
                final String jsonString = doc.get(CHAT_DATA).asStringOrNull();
                if (jsonString != null) {
                    jsonParser.parse(jsonString);
                    JsonObject obj = consumer.get();
                    doc.put(CHAT_DATA, obj);
                }
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        private BasicSuggestItem generateSavedMessagesChat(
            final String matchedText)
            throws JsonException
        {
            JsonMap data = new JsonMap(BasicContainerFactory.INSTANCE);
            JsonList emptyList = new JsonList(BasicContainerFactory.INSTANCE);
            JsonList members = new JsonList(BasicContainerFactory.INSTANCE);
            members.add(new JsonString(context.requestUserId()));
            JsonMap permissions = new JsonMap(BasicContainerFactory.INSTANCE);
            permissions.put("departments", emptyList);
            permissions.put("groups", emptyList);
            permissions.put("users", members);
            JsonList rights = new JsonList(BasicContainerFactory.INSTANCE);
            rights.add(new JsonString("read"));
            rights.add(new JsonString("write"));
            JsonMap roles = new JsonMap(BasicContainerFactory.INSTANCE);
            roles.put("admin", emptyList);
            String chatId = context.requestUserId() + '_' + context.requestUserId();

            data.put(CHAT_ID, new JsonString(chatId));
            data.put("create_timestamp", new JsonDouble(0));
            data.put("exclude", emptyList);
            data.put("geo_type", new JsonString("chat"));
            data.put("members", members);
            data.put("moderation_status", new JsonString("ok"));
            data.put("permissions", permissions);
            data.put("private", JsonBoolean.TRUE);
            data.put("restriction_status", new JsonString("active"));
            data.put("rights", rights);
            data.put("roles", roles);
            data.put("version", new JsonLong(0));

            JsonMap doc = new JsonMap(BasicContainerFactory.INSTANCE);
            doc.put(CHAT_DATA, data);
            doc.put("id", new JsonString(chatId));
            filterGetFields(doc);

            return new BasicSuggestItem(
                chatId,
                context.suggestType(),
                context.requestsTexts(),
                Collections.singletonList(
                    new SimpleEntry<>(NAME, matchedText)),
                SAVED_MESSAGES_SCORE,
                doc);
        }

        @Override
        public void completed(final Set<String> resources) {
            try {
//                System.err.println(context.request.text());
//                String request = context.requests.get(0).request();
                int count = 0;
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();

                final BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                final JsonParser jsonParser = new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(
                            consumer)));

                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    if (resourceId != null) {
                        ++count;
                        if (resources.contains(resourceId)) {
//                            String text = selectText(doc);
                            double score;
                            if (context.channelsSearch()) {
                                double requestScore = doc.getDouble("score", 0.0);
                                if (requestScore > 0) {
                                    requestScore = requestScore / 100;
                                }
                                score = 25.0 + requestScore;

                                if (score > 30.0) {
                                    score = 30.0;
                                }
                            } else {
                                double lmts = doc.getLong(ChatFields.LAST_MESSAGE_TIMESTAMP.stored(), 10L) / 1000.0;
                                double tsscore = BasicSuggestItem.scoreByMessageTs(lmts);
                                double luceneScore = doc.getDouble("#score", 0.0);
                                context.logger().info(
                                    "Lucene score " + luceneScore +  " tsscore" + tsscore + " for " + resourceId);
                                // 1.0 - bonus for chats rule
                                score = 30.0 + luceneScore + tsscore;
                            }

                            List<Map.Entry<String, String>> searchTexts =
                                searchTexts(doc);
                            filterGetFields(doc);
                            jsonReformat(doc, jsonParser, consumer);
                            BasicSuggestItem item =
                                new BasicSuggestItem(
                                    resourceId,
                                    context.suggestType(),
                                    context.requestsTexts(),
                                    searchTexts,
                                    score,
                                    doc);
                            items.add(item);
                        }
                    }
                }
                if (context.requestUserId() != null && !context.channelsSearch()) {
                    for (RequestInfo requestInfo: context.requests()) {
                        final String lowerCasedRequest =
                            requestInfo.request().toLowerCase(Locale.ROOT);
                        for (String saved: SAVED_MESSAGES) {
                            if (saved.startsWith(lowerCasedRequest)
                                || saved.contains(lowerCasedRequest))
                            {
                                items.add(generateSavedMessagesChat(saved));
                                break;
                            }
                        }
                    }
                }
                Collections.sort(items);
                if (items.size() >= context.length()
                    || count < requestedLength)
                {
                    callback.completed(items);
                } else {
                    execute(requestedLength << 1, context);
                }
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private void onChannelsExtracted(
        final ChatsSuggestRequestContext context,
        final int requestedLnegth,
        final Set<String> chats,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        if (context.suggestRequestContext().debug()) {
            context.session().logger().info(
                "User channels " + chats);
        } else {
            context.session().logger().info(
                "User channels size " + chats.size());
        }

        if (chats.isEmpty()) {
            callback.completed(Collections.emptyList());
            return;
        }

        String filter = context.session().params().getString(
            "channel_filter",
            null);

        StringBuilder queryFilter;
        if (filter != null) {
            queryFilter = new StringBuilder(filter);
            queryFilter.append(" AND ");
        } else {
            queryFilter = new StringBuilder();
        }

        queryFilter.append("chat_id:(");
        queryFilter.append(StringUtils.join(chats, ' '));
        queryFilter.append(')');

        execute(requestedLnegth, context, queryFilter.toString(), callback);
    }

    private class UserChatsCallback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final ChatsSuggestRequestContext context;
        private final int requestedLnegth;

        UserChatsCallback(
            final ChatsSuggestRequestContext context,
            final int requestedLnegth,
            final FutureCallback<? super List<SuggestItem>> callback)
        {
            super(callback);
            this.context = context;
            this.requestedLnegth = requestedLnegth;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                if (hits.size() == 0) {
                    callback.completed(Collections.emptyList());
                    return;
                }

                CollectionParser<String, Set<String>, Exception> chatsParser = UserChats.CHATS_SET_PARSER;
                if (context.namespaces() != null && context.namespaces().size() > 0) {
                    chatsParser = new CollectionParser<>(
                        LinkedHashSet::new,
                        (collection, sb) -> {
                            String chatId = sb.toString().trim();
                            String namespace = ChatUtils.namespaceFromChatId(chatId);
                            if (namespace != null && context.namespaces().contains(namespace)) {
                                collection.add(chatId);
                            }
                        },
                        '\n');
                }

                Set<String> chats =
                    hits.get(0).asMap().get(
                        UserChats.CHATS.stored(),
                        null,
                        chatsParser);

                if (chats == null || chats.isEmpty()) {
                    context.session().logger().info(
                        UserChats.CHATS.fieldName() + " is empty");
                    callback.completed(Collections.emptyList());
                    return;
                }

                Set<String> channels = new LinkedHashSet<>(chats.size());
                for (String chat: chats) {
                    if (chat.startsWith("1/")) {
                        channels.add(chat);
                    }
                }


                channelsCache.put(context.requestUserId(), channels);
                onChannelsExtracted(context, requestedLnegth, channels, callback);
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class ConcatCallback
        extends AbstractFilterFutureCallback<Map.Entry<List<SuggestItem>, List<SuggestItem>>, List<SuggestItem>>
    {
        private final ProxySession session;

        public ConcatCallback(
            final ProxySession session,
            final FutureCallback<? super List<SuggestItem>> callback)
        {
            super(callback);

            this.session = session;
        }

        @Override
        public void completed(final Map.Entry<List<SuggestItem>, List<SuggestItem>> entry) {
            List<SuggestItem> list =
                new ArrayList<>(
                    entry.getKey().size() + entry.getValue().size());

            session.logger().info("Chats got " + entry.getKey().size());
            session.logger().info("ChatsNewIndexation got " + entry.getValue().size());
            list.addAll(entry.getKey());
            list.addAll(entry.getValue());
            callback.completed(list);
        }
    }
}

