package ru.yandex.search.messenger.proxy.suggest.rules.messages;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import NMessengerProtocol.Message.TOutMessage;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.suggest.BasicSuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestItem;
import ru.yandex.search.messenger.proxy.suggest.SuggestRequestContext;
import ru.yandex.search.messenger.proxy.suggest.SuggestType;
import ru.yandex.search.messenger.proxy.suggest.rules.ProtoUtils;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestChatsFilterProvider;
import ru.yandex.search.messenger.proxy.suggest.rules.providers.SuggestRequestContextProvider;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.ShardPrefix;
import ru.yandex.search.request.util.FieldsTermsSupplierFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.string.UnhexStrings;

public class MessagesSuggestRule
    <T extends
        RequestProvider
        & SuggestRequestContextProvider
        & SuggestChatsFilterProvider>
    implements SearchRule<T, List<SuggestItem>>
{
    private static final int MIN_REQUEST_LEN = 2;
    private static final String CHAT_ID = "message_chat_id";
    private static final String MESSAGE_TEXT = "message_text";
    private static final String MESSAGE_POLL_TITLE = "message_poll_title";
    private static final String MESSAGE_POLL_ANSWERS = "message_poll_answers";
//    private static final String MESSAGE_FILENAME = "message_filename";
//    private static final String MESSAGE_RCA_SNIPPET = "message_first_link_snippet";
//    private static final String MESSAGE_RCA_TITLE = "message_first_link_title";
//    private static final String MESSAGE_LINKS = "message_links";
    private static final String MESSAGE_DATA = "message_data";
    private static final String MESSAGE_TIMESTAMP = "message_timestamp";
    private static final String PREFIX = "&prefix=";
    private static final String[] SEARCH_FIELDS_PREFIXED = {
        MESSAGE_TEXT + "_p",
        MESSAGE_POLL_TITLE + "_p",
        MESSAGE_POLL_ANSWERS + "_p"
    };

    private static final String[] SEARCH_FIELDS = {
        MESSAGE_TEXT,
        MESSAGE_POLL_TITLE,
        MESSAGE_POLL_ANSWERS
    };

    private static final String[] MATCH_FIELDS = {
        MESSAGE_TEXT,
        MESSAGE_POLL_TITLE,
        MESSAGE_POLL_ANSWERS
    };
    private static final FieldsTermsSupplierFactory PREFIXED_FIELDS =
        new FieldsTermsSupplierFactory(SEARCH_FIELDS_PREFIXED);
    private static final FieldsTermsSupplierFactory GLOBAL_FIELDS =
        new FieldsTermsSupplierFactory(SEARCH_FIELDS);
    private static final FieldsTermsSupplierFactory MESSAGE_LINKS_FIELDS =
        new FieldsTermsSupplierFactory("message_links", "message_first_link_snippet", "message_first_link_title");
    private static final FieldsTermsSupplierFactory MESSAGE_FILE_FIELDS =
        new FieldsTermsSupplierFactory("message_filename_p","message_filename_tokenized_p");
    private static final int MIN_LENGTH = 10;
    private static final String RESOURCE_ID = "message_id";

    private final long failoverDelay;
    private final boolean localityShuffle;
    private final Moxy moxy;
    private INum[] globalShards = new INum[0];
    private int searchMapVersion = -1;

    public MessagesSuggestRule(final Moxy moxy) {
        this.moxy = moxy;
        failoverDelay = moxy.config().messagesSuggestFailoverDelay();
        localityShuffle = moxy.config().messagesSuggestLocalityShuffle();
        calcGlobalShards();
    }

    private void calcGlobalShards() {
        int searchMapVersion = moxy.searchMap().version();
        if (searchMapVersion == this.searchMapVersion) {
            return;
        }
        SearchMapRow row =
            moxy.searchMap().row(moxy.config().messagesService());
        if (row == null) {
            globalShards = new INum[0];
        } else {
            Map<Integer, INum> mergeMap = new HashMap<>();
            for (int i = 0; i < SearchMap.SHARDS_COUNT; i++) {
                SearchMapShard shard = row.get(i);
                if (shard != null) {
                    Prefix prefix = new ShardPrefix(i);
                    User user = new User(
                        moxy.config().messagesService(),
                        prefix);
                    mergeMap.putIfAbsent(shard.iNum(), new INum(shard, user));
                }
            }
            globalShards = mergeMap.values().toArray(new INum[0]);
        }
        this.searchMapVersion = searchMapVersion;
    }

    //CSOFF: ReturnCount
    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<SuggestItem>> callback)
        throws HttpException
    {
        SuggestRequestContext context = input.suggestRequestContext();
        if (input.chatsFilter().empty()) {
            context.logger().info("Empty chats list");
            callback.completed(Collections.emptyList());
            return;
        }
        String request = input.request();
        if (context.hadRequest(SuggestType.MESSAGES, request)) {
            context.logger().info("Request: " + request
                + "has already been executed. Skipping");
            callback.completed(Collections.emptyList());
            return;
        }
        if (request.length() < MIN_REQUEST_LEN) {
            context.logger().info("Request:  " + request
                + " to small for messages suggest. Skipping");
            callback.completed(Collections.emptyList());
            return;
        }
        execute(
            Math.max(MIN_LENGTH, context.length() << 1),
            new MessagesSuggestRequestContext(
                context,
                request,
                SearchRequestText.parseSuggest(input.request(), false),
                input.chatsFilter(),
                failoverDelay,
                localityShuffle,
                globalShards,
                callback));
    }
    //CSON: ReturnCount

    public static void execute(
        final int requestedLength,
        final MessagesSuggestRequestContext context)
        throws HttpException
    {
        StringBuilder sb = new StringBuilder("");
        SearchRequestText request = context.requestText();
        if (request.isEmpty() && request.text().length() > 0) {
            context.logger().info("No human words in  Request: " + request
                + " .Skipping ");
            context.callback().completed(Collections.emptyList());
            return;
        }
        if (context.queryFilter() != null) {
            sb.append('(');
            sb.append(context.queryFilter());
            sb.append(')');
        }
        FieldsTermsSupplierFactory fieldsFactory;
        if (context.mediaTypeFilter() != null) {
            if (context.mediaTypeFilter() == MessagesSuggestRequestContext.ChatMediaFilter.LINK) {
                fieldsFactory = MESSAGE_LINKS_FIELDS;
            } else {
                if (context.mediaTypeFilter() == MessagesSuggestRequestContext.ChatMediaFilter.FILE_MESSAGE) {
                    fieldsFactory = MESSAGE_FILE_FIELDS;
                } else {
                    throw new BadRequestException("media type filter should be files/links");
                }
            }
        } else {
            if (context.globalSearch()) {
                fieldsFactory = GLOBAL_FIELDS;
            } else {
                fieldsFactory = PREFIXED_FIELDS;
            }
        }
        if (request.hasWords()) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            sb.append('(');
            request.fieldsQuery(
                sb,
                fieldsFactory,
                 ") AND (");
            sb.append(')');
        } else if (sb.length() == 0) {
            context.logger().info("Empty token list for Request: " + request
                + " .Skipping");
            context.callback().completed(Collections.emptyList());
            return;
        }
//        request.negationsQuery(sb, NAME_FIELD);
        LinkedHashSet<String> get = new LinkedHashSet<>();
        get.add(RESOURCE_ID);
        get.add(MESSAGE_TEXT);
        get.add(MESSAGE_POLL_TITLE);
        get.add(MESSAGE_POLL_ANSWERS);
        get.add(CHAT_ID);
        get.add(MESSAGE_TIMESTAMP);
        get.addAll(context.getFields());
        String pruningField = "message_hour_p";
        if (context.globalSearch()) {
            pruningField = "message_hour";
        }
        QueryConstructor query =
            new QueryConstructor(
                "/search?IO_PRIO=0&json-type=dollar"
                + "&sync-searcher=false"
                + "&skip-nulls"
                + "&no_scorer=lucene&sort=message_timestamp"
                + "&collector=pruning(" + pruningField + ')'
                + "&dp=fallback(message_parent_id,message_id+message_id)");
//        User user = context.user();
        query.append("get", StringUtils.join(get, ','));
        query.append("service", context.service());
        query.append("text", new String(sb));
        query.append("length", requestedLength);
//        query.append("group", "id");
//        query.append("group", "multi(message_id,message_timestamp)");
        if (context.dps() != null && context.dps().size() > 0) {
            for (String dp: context.dps()) {
                query.append("dp", dp);
            }
        }
        if (context.postfilters() != null && context.postfilters().size() > 0) {
            for (String pf: context.postfilters()) {
                query.append("postfilter", pf);
            }
        }
        String commonUri = query.toString();
        MultiFutureCallback<List<SuggestItem>> multiCallback =
            new MultiFutureCallback<>(
                new MergeResultsCallback(context));
//                new MergeResultsCallback(context, requestedLength));
        for (ShardContext subContext: context.subContexts()) {
            context.logger().info("running search for shard: "
                + subContext.shard());
            String uri;
            if (context.globalSearch()) {
                uri = commonUri;
            } else {
                StringBuilder uriSb =
                    new StringBuilder(commonUri.length() + subContext.chats().size() * 25 + 100);
                uriSb.append(commonUri);
                for (Map.Entry<String, Long> entry: subContext.chats().entrySet()) {
                    uriSb.append(PREFIX);
                    uriSb.append(entry.getKey());

                    if (entry.getValue() > 0) {
                        uriSb.append("&dp=chat_ts_filter(");
                        uriSb.append(entry.getKey());
                        uriSb.append(',');
                        uriSb.append(entry.getValue());
                        uriSb.append(')');
                    }
                }
                uri = uriSb.toString();
            }
            context.proxy().sequentialRequest(
                context.session(),
                subContext,
                new BasicAsyncRequestProducerGenerator(uri),
                context.failoverDelay(),
                context.localityShuffle(),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                new Callback(
                    context,
                    multiCallback.newCallback()));
//                    requestedLength));
        }
        multiCallback.done();
    }

    private static class MergeResultsCallback
        extends AbstractFilterFutureCallback<
            List<List<SuggestItem>>,
            List<SuggestItem>>
    {
//        private final MessagesSuggestRequestContext context;
//        private final int requestedLength;

        MergeResultsCallback(
            final MessagesSuggestRequestContext context)
//            final int requestedLength)
        {
            super(context.callback());
//            this.context = context;
//            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final List<List<SuggestItem>> result) {
//            System.err.println("Merged: " + result.size());
            Map<String, SuggestItem> mergeMap = new HashMap<>();
            for (List<SuggestItem> subItems: result) {
//                System.err.println("Merged sub: " + subItems.size());
                for (SuggestItem item: subItems) {
//                    System.err.println("Merged item: " + item);
                    mergeMap.put(item.id(), item);
                }
            }
            ArrayList<SuggestItem> merged = new ArrayList<>(mergeMap.size());
            merged.addAll(mergeMap.values());
            merged.sort((i1, i2) -> Double.compare(i2.score(), i1.score()));
//            System.err.println("Merged list: " + merged.size());
//            System.err.println("Merged list1: " + merged);
            callback.completed(merged);
        }
    }

    @SuppressWarnings("HidingField")
    private static class Callback
        extends AbstractFilterFutureCallback<JsonObject, List<SuggestItem>>
    {
        private final MessagesSuggestRequestContext context;
        private final FutureCallback<List<SuggestItem>> callback;
//        private final int requestedLength;

        Callback(
            final MessagesSuggestRequestContext context,
            final FutureCallback<List<SuggestItem>> callback)
//            final int requestedLength)
        {
            super(callback);
            this.callback = callback;
            this.context = context;
//            this.requestedLength = requestedLength;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                context.logger().info("Before filter:" + hits.size());
//                System.err.println(
//                    "Lucene response: "
//                    + JsonType.HUMAN_READABLE.toString(hits));
                context.proxy().filterMessagesResources(
                    context,
                    hits,
                    context.chatsFilter(),
                    new FilteredResourcesCallback(
                        callback,
                        context,
//                        requestedLength,
                        hits));
            } catch (HttpException | JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }

    private static class FilteredResourcesCallback
        extends AbstractFilterFutureCallback<Set<String>, List<SuggestItem>>
    {
        private final MessagesSuggestRequestContext context;
//        private final int requestedLength;
        private final JsonList hits;

        FilteredResourcesCallback(
            final FutureCallback<List<SuggestItem>> callback,
            final MessagesSuggestRequestContext context,
//            final int requestedLength,
            final JsonList hits)
        {
            super(callback);
            this.context = context;
//            this.requestedLength = requestedLength;
            this.hits = hits;
        }

        private void filterGetFields(final JsonMap doc) {
            Iterator<Map.Entry<String, JsonObject>> docIter =
                doc.entrySet().iterator();
            while (docIter.hasNext()) {
                String field = docIter.next().getKey();
                if (!context.getFields().contains(field)) {
                    docIter.remove();
                }
            }
        }

        private void base64Reformat(final JsonMap doc) {
            try {
                final String hexString = doc.get(MESSAGE_DATA).asStringOrNull();
                if (hexString != null) {
                    byte[] data = UnhexStrings.unhex(hexString);
                    Base64Encoder encoder = new Base64Encoder();
                    encoder.process(data);
                    doc.put(MESSAGE_DATA, new JsonString(encoder.toString()));
                }
                commonReformat(doc);
            } catch (JsonException e) { // skip, obj is null
            }
        }

        private void jsonReformat(final JsonMap doc) {
            try {
                final String hexString = doc.get(MESSAGE_DATA).asStringOrNull();
                if (hexString != null) {
                    byte[] data = UnhexStrings.unhex(hexString);
                    TOutMessage message = TOutMessage.parseFrom(data);
                    doc.put(MESSAGE_DATA, ProtoUtils.protoToJson(message));
                }
                commonReformat(doc);
            } catch (InvalidProtocolBufferException | JsonException e) {
                context.logger().log(
                    Level.WARNING,
                    "Can't reformat proto to json",
                    e);
            }
        }

        private void commonReformat(final JsonMap doc) throws JsonBadCastException {
            final String links = doc.get("message_links").asStringOrNull();
            if (links != null) {
                JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
                for (String link: links.split("\\n")) {
                    list.add(new JsonString(link));
                }
                doc.put("message_links", list);
            }
            final String pollAnswers = doc.get("message_poll_answers").asStringOrNull();
            if (pollAnswers != null) {
                JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
                for (String answer: pollAnswers.split("\\n")) {
                    list.add(new JsonString(answer));
                }
                doc.put("message_poll_answers", list);
            }
        }

        private List<Map.Entry<String, String>> searchTexts(final JsonMap doc)
            throws JsonException
        {
            final List<Map.Entry<String, String>> texts =
                new ArrayList<>(MATCH_FIELDS.length);
            for (String field: MATCH_FIELDS) {
                String text = doc.get(field).asStringOrNull();
                if (text != null) {
                    texts.add(new SimpleEntry<>(field, text));
                }
            }
            return texts;
        }

        @Override
        public void completed(final Set<String> resources) {
            try {
                //System.err.println(context.requestText().text());
                String request = context.requestText().text();
                List<SuggestItem> items = new ArrayList<>(resources.size());
                Iterator<JsonObject> iter = hits.iterator();

                while (iter.hasNext()) {
                    JsonMap doc = iter.next().asMap();
                    String resourceId = doc.get(RESOURCE_ID).asStringOrNull();
                    if (resourceId != null) {
//                        System.err.println(request
//                            + ": resource_id: " + resourceId);
                        if (resources.contains(resourceId)) {
//                            System.err.println(request
//                                + ": resource_id.ok");
//                            String text = selectText(doc);
                            // -100 messages debonus
                            double score = -100.0 + BasicSuggestItem.scoreByMessageMicroTs(doc.get(MESSAGE_TIMESTAMP).asDouble());
                            System.err.println(request
                                + ": score: " + score);
                            List<Map.Entry<String, String>> searchTexts =
                                searchTexts(doc);
//                            System.err.println(request
//                                + ": searchTexts: " + searchTexts);
                            filterGetFields(doc);
                            if (context.base64Out()) {
                                base64Reformat(doc);
                            } else if (context.jsonOut()) {
                                jsonReformat(doc);
                            }
                            BasicSuggestItem item =
                                new BasicSuggestItem(
                                    resourceId,
                                    SuggestType.MESSAGES,
                                    request,
                                    searchTexts,
                                    score,
                                    doc);
                            items.add(item);
//                            System.err.println(request
//                                + ": items.sofar: " + items.size());
                        }
                    }
                }
//                if (items.size() >= context.length()
//                    || count < requestedLength)
//                {
//                System.err.println(request
//                    + ": items.final: " + items.size());

                context.logger().info("After filter:" + items.size());
                callback.completed(items);
//                } else {
//                    execute(requestedLength << 1, context);
//                }
//            } catch (HttpException | JsonException e) {
            } catch (JsonException e) {
                failed(new ServiceUnavailableException(e));
            }
        }
    }
}

