package ru.yandex.search.messenger.proxy.topposts;

import java.io.IOException;

import NMessengerProtocol.Client;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.messenger.proxy.suggest.rules.ProtoUtils;
import ru.yandex.util.string.UnhexStrings;

public class Message implements Comparable<Message> {
    private static final int MIN_IMAGE_SIZE = 240;
    private static final String SNIPPET = "snippet";
    private static final String MESSAGE_DATA = "message_data";

    private NMessengerProtocol.Message.TOutMessage message;
    private final long seqNo;
    private final long timestamp;
    private final boolean hasImage;
    private final int imageWidth;
    private final int imageHeight;
    private final String imageUrl;
    private final String snippet;

    public Message(
        final JsonMap json,
        final SimpleJsonParser parser)
        throws JsonException
    {
        String data = json.get(MessageFields.DATA.stored()).asStringOrNull();
        seqNo = json.getLong(MessageFields.SEQ_NO.stored(), 0L);
        timestamp = json.getLong(MessageFields.TIMESTAMP.stored(), 0L);
        if (data != null) {
            try {
                ReusableCodedInputStream rcis =
                    TopPostsHandler.getTlsCodedInputStream(data.length() >> 1);
                byte[] bytes = rcis.array();
                UnhexStrings.unhex(data, bytes);
//                    message = TOutMessage.parseFrom(bytes);
//                    ByteString bs = ByteString.copyFrom(bytes);
//                    message = TOutMessage.parseFrom(bs);
                message = NMessengerProtocol.Message.TOutMessage.parseFrom(rcis.cis());
            } catch (IOException e) {
                throw new JsonException(e);
            }
        } else {
            message = null;
        }
        boolean hasImage = false;
        int imageWidth = 0;
        int imageHeight = 0;
        String imageUrl = null;
        String snippet = null;
        String rcaData = json.get(MessageFields.RCA_DATA.stored()).asStringOrNull();
        if (rcaData != null) {
            JsonObject rcaObject = parser.parse(rcaData);
            JsonMap rcaMap = rcaObject.asMap();
            JsonMap mainImage = rcaMap.getMapOrNull("main_image");
            if (mainImage != null) {
                Integer height = mainImage.getInt("height", null);
                Integer width = mainImage.getInt("width", null);
                if (height != null && width != null
                    && Math.min(height, width) > MIN_IMAGE_SIZE)
                {
                    hasImage = true;
                    imageUrl = mainImage.getString("src");
                    imageWidth = width;
                    imageHeight = height;
                    snippet = rcaMap.get(SNIPPET).asStringOrNull();
                }
            }
        }
        this.hasImage = hasImage;
        this.imageWidth = imageWidth;
        this.imageHeight = imageHeight;
        this.imageUrl = imageUrl;
        this.snippet = snippet;
    }

    @Override
    public int hashCode() {
        return Long.hashCode(seqNo);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof Message) {
            Message other = (Message) o;
            return other.seqNo == seqNo;
        }
        return false;
    }

    @Override
    public int compareTo(final Message o) {
        return Long.compare(o.seqNo, seqNo);
    }

    public Client.TServerMessage serverMessageProto() {
        if (message != null) {
            return message.getServerMessage();
        } else {
            return null;
        }
    }

    public void writeJson(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("has_image");
        writer.value(hasImage);
        if (hasImage) {
            writer.key("image_height");
            writer.value(imageHeight);
            writer.key("image_width");
            writer.value(imageWidth);
            writer.key("image_url");
            writer.value(imageUrl);
            writer.key(SNIPPET);
            writer.value(snippet);
        }
        writer.key(MESSAGE_DATA);
        ProtoUtils.protoToJson(message).writeValue(writer);
        writer.endObject();
    }

    public NMessengerProtocol.Message.TOutMessage message() {
        return message;
    }

    public long seqNo() {
        return seqNo;
    }

    public long timestamp() {
        return timestamp;
    }

    public boolean hasImage() {
        return hasImage;
    }

    public int imageWidth() {
        return imageWidth;
    }

    public int imageHeight() {
        return imageHeight;
    }

    public String imageUrl() {
        return imageUrl;
    }

    public String snippet() {
        return snippet;
    }
}
