package ru.yandex.search.messenger.proxy.topposts;

import java.util.Collection;
import java.util.Map;

import org.apache.http.HttpException;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.util.string.StringUtils;

public class ShardMessagesRequestContext extends RequestContext {
    private static final String GET =
        MessageFields.ID.stored() + ','
            + MessageFields.CHAT_ID.stored() + ','
            + MessageFields.DATA.stored() + ','
            + MessageFields.SEQ_NO.stored() + ','
            + MessageFields.TIMESTAMP.stored() + ','
            + MessageFields.RCA_DATA.stored();

    private final User user;
    private final Map<String, JsonMap> chats;
    private final boolean online;

    //CSOFF: ParameterNumber
    public ShardMessagesRequestContext(
        final TopPostsRequestContext context,
        final boolean online,
        final Map<String, JsonMap> chats)
        throws HttpException
    {
        super(context);
        this.user = new User(context.config().messagesService(), new StringPrefix(chats.keySet().iterator().next()));
        this.chats = chats;
        this.online = online;
    }
    //CSON: ParameterNumber

    @Override
    protected String service() {
        return topContext().config().messagesService();
    }

    public Collection<String> chatIds() {
        return chats.keySet();
    }

    public Map<String, JsonMap> chats() {
        return chats;
    }

    public QueryConstructor createQuery()
        throws HttpException
    {
        return createQuery(topContext());
    }

    @Override
    protected QueryConstructor createQuery(
        final TopPostsRequestContext context)
        throws HttpException
    {
        String get = GET;
        if (context.get() != null) {
            get += ',' + context.get();
        }


        QueryConstructor query = new QueryConstructor(
            "/search?json-type=dollar&IO_PRIO=0");
        query.append("sort", context.sort());
        query.append("asc", context.asc());
        query.append("get", get);
        query.append("text", searchText(context));
        if (online) {
            for (String chatId: chats.keySet()) {
                query.append("prefix", chatId);
            }

            query.append("group", MessageFields.CHAT_ID.stored());
        }

        query.append("collector", "sorted");

        long timeLimit = System.currentTimeMillis()
            - (context.daysDepth() * TopPostsHandler.MILLIS_PER_DAY);
        timeLimit *= TopPostsHandler.MS_PER_MICRO;
        query.append(
            "postfilter",
            MessageFields.TIMESTAMP.stored() + " >= " + timeLimit);
        if (context.filterEmptyText()) {
            query.append("dp", "equals(message_text,,  text_empty)");
            query.append("postfilter", "text_empty != 1");
        }
        return query;
    }

    private String searchText(
        final TopPostsRequestContext context)
    {
        final StringBuilder sb = new StringBuilder();
        long time = System.currentTimeMillis();
        sb.append("message_day_p:(");
        sb.append(time * TopPostsHandler.MS_PER_MICRO);
        char sep = ' ';
        //we should select daysDepth + 1 days and than trim then by
        //docprocessor to emulate 24h limit
        for (int i = 0; i < context.daysDepth(); i++) {
            sb.append(sep);
            time -= TopPostsHandler.MILLIS_PER_DAY;
            sb.append(time * TopPostsHandler.MS_PER_MICRO);
        }
        sb.append(')');
        if (context.types() != null) {
            sb.append(" AND type_p:(");
            sb.append(StringUtils.join(context.types(), ' '));
            sb.append(')');
        }
        if (context.moderationActions() != null) {
            sb.append(" AND message_moderation_action:(");
            sb.append(StringUtils.join(context.moderationActions(), ' '));
            sb.append(')');
        }
        return new String(sb);
    }

    @Override
    protected User user() {
        return user;
    }

    public PrefixedLogger logger() {
        return topContext().logger();
    }
}
