package ru.yandex.search.messenger.proxy.topposts;

import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.LooseProcessors;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.search.messenger.ChatFields;
import ru.yandex.search.messenger.UserChats;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class TopPostsHandler implements ProxyRequestHandler {
    public static final long MS_PER_MICRO = 1000;
    public static final long MILLIS_PER_DAY = 86400 * 1000;

    private static final String SERVICE = "service";
    private static final String HITS_ARRAY = "hitsArray";
    private static final String MESSAGE_CHAT_ID = "message_chat_id";

    private static final ConcurrentLinkedQueue<SimpleJsonParser> JSON_PARSERS =
        new ConcurrentLinkedQueue<>();

    private static final ThreadLocal<ReusableCodedInputStream> CODED_INPUT_TLS =
        ThreadLocal.withInitial(
            () -> new ReusableCodedInputStream());

    private final Moxy moxy;
    private final String messagesService;
    private final ThreadPoolExecutor executor;
    private final TopPostsUpdater updater;
    private final Set<String> channelsFilter;

    public TopPostsHandler(final Moxy moxy) throws IOException {
        this.moxy = moxy;
        LooseProcessors.init();
        messagesService = moxy.config().messagesService();
        executor =
            new ThreadPoolExecutor(
                moxy.config().workers(),
                moxy.config().workers(),
                1,
                TimeUnit.DAYS,
                new LinkedBlockingQueue<>());

        if (moxy.config().topPostConfig().hardcodedChannels() != null) {
            try {
                JsonObject root = TypesafeValueContentHandler.parse(
                    new FileReader(moxy.config().topPostConfig().hardcodedChannels(),
                        StandardCharsets.UTF_8));
                JsonList chatList = root.asMap().getMap("recommended_chats_config").getList("top_weighted");
                Set<String> channels = new LinkedHashSet<>();
                for (JsonObject channelObj: chatList) {
                    channels.add(channelObj.asMap().getString("chat_id"));
                }

                channelsFilter = Collections.unmodifiableSet(channels);
            } catch (JsonException e) {
                throw new IOException(e);
            }
        } else {
            channelsFilter = null;
        }

        if (!moxy.config().topPostConfig().cachedParams().isEmpty()) {
            updater = new TopPostsUpdater(moxy, channelsFilter);
        } else {
            updater = null;
        }
    }

    public static void applyTimestampFilter(
        final String field,
        final StringBuilder sb,
        final int daysDepth)
    {
        long time = System.currentTimeMillis();
        sb.append(field);
        sb.append(":(");
        sb.append(time * MS_PER_MICRO);
        char sep = ' ';
        //we should select daysDepth + 1 days and than trim then by
        //docprocessor to emulate 24h limit
        for (int i = 0; i < daysDepth; i++) {
            sb.append(sep);
            time -= MILLIS_PER_DAY;
            sb.append(time * MS_PER_MICRO);
        }
        sb.append(')');
    }

    public static void applyTimestampPostFilter(
        final QueryConstructor query,
        final String field,
        final int daysDepth)
        throws BadRequestException
    {
        long timeLimit = System.currentTimeMillis()
            - (daysDepth * MILLIS_PER_DAY);
        timeLimit *= MS_PER_MICRO;
        query.append(
            "postfilter",
            field + " >= " + timeLimit);
    }

    public static ReusableCodedInputStream getTlsCodedInputStream(
        final int size)
        throws InvalidProtocolBufferException
    {
        ReusableCodedInputStream rcis = CODED_INPUT_TLS.get();
        rcis.reset();
        rcis.checkSize(size);
        return rcis;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        String uri = session.request().getRequestLine().getUri();
        final TopPostsRequestContext topPostsContext =
            new TopPostsRequestContext(session.params(), session.logger(), moxy.config());
        final ChatsRequestContext chatsContext =
            new ChatsRequestContext(topPostsContext);
        TopPostsPrinter printer = new TopPostsPrinter(topPostsContext, session);

        CgiParams params = new CgiParams(session.params());
        params.remove("proto");
        params.remove("chat-ids");
        params.remove("uid");

        TopPostsUpdater.CachedResponse cachedResponse = null;
        if (updater != null) {
            cachedResponse = updater.cache(params);
        }

        if (cachedResponse != null) {
            if (!topPostsContext.chatIds().isEmpty()) {
                Set<String> diff = new LinkedHashSet<>(topPostsContext.chatIds());
                diff.removeAll(cachedResponse.chats().keySet());
                if (diff.size() > 0) {
                    session.logger().info(
                        "We not checked chats: " + diff.size() + " " + diff);
                }
            }

            session.logger().info(
                "From cache, freshness "
                    + TimeUnit.MILLISECONDS.toSeconds(
                    System.currentTimeMillis() - cachedResponse.ts()));

            if (topPostsContext.uid() != null) {
                sendRequest(
                    new ChatsListByUidContext(topPostsContext),
                    session,
                    new ChatsWithCachedResponse(
                        topPostsContext,
                        printer,
                        cachedResponse.chats()));
                return;
            }

            printer.completed(cachedResponse.chats().values());
            return;
        }

        sendRequest(
            chatsContext,
            session,
            new ChatsCallback(
                session,
                printer,
                topPostsContext));
    }

    private static class ChatsWithCachedResponse
        extends AbstractFilterFutureCallback<JsonObject, Collection<Chat>>
    {
        private final Map<String, Chat> cached;
        private final TopPostsRequestContext context;

        public ChatsWithCachedResponse(
            final TopPostsRequestContext context,
            final FutureCallback<? super Collection<Chat>> callback,
            final Map<String, Chat> cached)
        {
            super(callback);
            // immutable !
            this.cached = cached;
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                List<Chat> resultList = new ArrayList<>(cached.size());

                if (hits.size() <= 0) {
                    context.logger().info("User chats on morda not found");
                    callback.completed(cached.values());
                    return;
                }

                JsonMap hit = hits.get(0).asMap();
                Set<String> userChats =
                    hit.get(
                        UserChats.CHATS.stored(),
                        Collections.emptySet(),
                        UserChats.CHATS_SET_PARSER);
                context.logger().info("User chats found: " + userChats.size());

                for (Map.Entry<String, Chat> entry: cached.entrySet()) {
                    if (entry.getValue().messages().size() == 0) {
                        continue;
                    }

                    if (userChats.contains(entry.getKey())) {
                        resultList.add(new Chat(entry.getValue(), true));
                    } else {
                        resultList.add(entry.getValue());
                    }
                }

                callback.completed(resultList);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private void procceedMessagesRequest(
        final ProxySession session,
        final TopPostsRequestContext topPostsContext,
        final TopPostsPrinter printer,
        final JsonObject chatsResponse)
        throws HttpException
    {
        Map<String, JsonMap> chats;
        try {
            JsonList hits = chatsResponse.asMap().getList("hitsArray");
            chats = new LinkedHashMap<>(hits.size());

            for (JsonObject hitObj: hits) {
                JsonMap hit = hitObj.asMap();
                String chatId = hit.getOrNull(ChatFields.ID.stored());
                if (chatId == null) {
                    topPostsContext.logger().warning("No chat id in " + JsonType.NORMAL.toString(hit));
                    continue;
                }

                chats.put(chatId, hit);
            }
        } catch (JsonException je) {
            printer.failed(je);
            return;
        }

        MergeCallback mergeCallback = new MergeCallback(printer, topPostsContext);

        if (topPostsContext.skipMessagesRequest()) {
            Map<String, Chat> chatMap = new LinkedHashMap<>(chats.size());
            for (Map.Entry<String, JsonMap> entry: chats.entrySet()) {
                chatMap.put(entry.getKey(), new Chat(entry.getValue(), Collections.emptyList()));
            }

            mergeCallback.completed(Collections.singletonList(chatMap));
        } else {
            final MessagesRequestContext messagesContext =
                new MessagesRequestContext(session, chats, topPostsContext);

            MultiFutureCallback<Map<String, Chat>> multiCallback =
                new MultiFutureCallback<>(mergeCallback);
            for (RequestContext context: messagesContext.subContexts()) {
                sendMessagesRequest(
                    context,
                    session,
                    chats,
                    multiCallback.newCallback());
            }
            multiCallback.done();
        }
    }

    public void sendMessagesRequest(
        final RequestContext context,
        final ProxySession session,
        final Map<String, JsonMap> chats,
        final FutureCallback<Map<String, Chat>> callback)
        throws HttpException
    {
        sendRequest(
            context,
            session,
            new JsonToTopMessagesCallback(
                callback,
                chats,
                context,
                session));
    }

    public void sendRequest(
        final RequestContext context,
        final ProxySession session,
        final FutureCallback<JsonObject> callback)
        throws HttpException
    {
        AsyncClient client = moxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                context.user(),
                null,
                context.topContext().allowLaggingHosts(),
                client,
                session.logger());
        QueryConstructor query = context.query();
        query.append(SERVICE, context.service());
        moxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            context.topContext().failoverDelay(),
            context.topContext().localityShuffle(),
//            JsonAsyncTypesafeDomConsumerFactory.OK,
            AsyncStringConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new JsonParseCallback(
                callback,
                executor,
                session,
                context.getClass().getName()));
    }

    @Override
    public String toString() {
        return "Messenger top messages for selected chats handler: "
            + "https://wiki.yandex-team.ru/ps/documentation/"
            + "moxy#tophosts";
    }

    private class MessagesRequestContext {
        private List<ShardMessagesRequestContext> subContexts;

        MessagesRequestContext(
            final ProxySession session,
            final Map<String, JsonMap> chatsData,
            final TopPostsRequestContext context)
            throws HttpException
        {
            Map<SearchMapShard, Map<String, JsonMap>> shardMap = new HashMap<>();
            Map<SearchMapShard, User> userMap = new HashMap<>();
            for (Map.Entry<String, JsonMap> entry: chatsData.entrySet()) {
                String chatId = entry.getKey();
                Prefix prefix = new StringPrefix(chatId);
                User user = new User(messagesService, prefix);
                SearchMapShard shard = moxy.searchMap().apply(user);
                Map<String, JsonMap> chats = shardMap.get(shard);
                if (chats == null) {
                    chats = new LinkedHashMap<>(2);
                    shardMap.put(shard, chats);
                    userMap.put(shard, user);
                }
                chats.put(chatId, entry.getValue());
            }
            subContexts = new ArrayList<>(shardMap.size());
            for (Map.Entry<SearchMapShard, Map<String, JsonMap>> entry
                : shardMap.entrySet())
            {
                User user = userMap.get(entry.getKey());
                session.logger().info("SubContext: prefixed, user: "
                    + user + ", shard:  " + entry.getKey());
                subContexts.add(
                    new ShardMessagesRequestContext(
                        context,
                        true,
                        entry.getValue()));
            }
        }

        public List<ShardMessagesRequestContext> subContexts() {
            return subContexts;
        }
    }

    public static SimpleJsonParser getJsonParser() {
        SimpleJsonParser parser = JSON_PARSERS.poll();
        if (parser == null) {
            parser = new SimpleJsonParser();
        }
        return parser;
    }

    public static void freeJsonParser(final SimpleJsonParser parser) {
        JSON_PARSERS.add(parser);
    }

    private class ChatsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final TopPostsRequestContext context;
        private final TopPostsPrinter printer;

        ChatsCallback(
            final ProxySession session,
            final TopPostsPrinter printer,
            final TopPostsRequestContext context)
            throws HttpException
        {
            super(session);
            this.context = context;
            this.printer = printer;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                Set<String> chatIds = new HashSet<>();
                for (JsonObject o: hits) {
                    JsonMap hit = o.asMap();
                    String chatId = hit.getString("chat_id");
                    if (channelsFilter != null && !channelsFilter.contains(chatId)) {
                        context.logger().info("Channel filtered by hardcoded list");
                        continue;
                    }
                    chatIds.add(chatId);
                }
                if (chatIds.size() == 0) {
                    String msg =
                        "No chats found with show_on_morda flag";
                    session.logger().severe(msg);
                    session.response(HttpStatus.SC_NOT_FOUND, msg);
                } else {
                    context.chatIds().addAll(chatIds);
                    procceedMessagesRequest(
                        session,
                        context,
                        printer,
                        result);
                }
            } catch (HttpException| JsonException e) {
                failed(e);
            }
        }
    }

    private final class JsonToTopMessagesCallback
        extends AbstractFilterFutureCallback<JsonObject, Map<String, Chat>>
    {
        private final RequestContext context;
        private final ProxySession session;
        private final Map<String, JsonMap> chatsData;

        JsonToTopMessagesCallback(
            final FutureCallback<Map<String, Chat>> callback,
            final Map<String, JsonMap> chats,
            final RequestContext context,
            final ProxySession session)
        {
            super(callback);
            this.context = context;
            this.session = session;
            this.chatsData = chats;
        }

        @Override
        public void completed(final JsonObject response) {
            SimpleJsonParser jsonParser = getJsonParser();
            long startTime = System.currentTimeMillis();
            try {
                JsonList shardHits =
                    response.asMap().getList(HITS_ARRAY);
                Map<String, List<Message>> topMessages = new LinkedHashMap<>(shardHits.size());

                for (JsonObject obj: shardHits) {
                    JsonMap hit = obj.asMap();
                    Message message = new Message(hit, jsonParser);
                    String chatId = hit.getOrNull(MESSAGE_CHAT_ID);
                    if (chatId == null) {
                        continue;
                    }

                    topMessages.computeIfAbsent(
                        chatId,
                        (k) -> new ArrayList<>())
                        .add(message);

                    JsonList mergedList = hit.getListOrNull("merged_docs");
                    if (mergedList != null) {
                        for (JsonObject merged: mergedList) {
                            message = new Message(
                                merged.asMap(),
                                jsonParser);
                            topMessages.computeIfAbsent(
                                chatId,
                                (k) -> new ArrayList<>())
                                .add(message);
                        }
                    }
                }

                Map<String, Chat> chats = new LinkedHashMap<>();
                for (Map.Entry<String, List<Message>> entry: topMessages.entrySet()) {
                    List<Message> messages = entry.getValue();
                    Collections.sort(messages);
                    messages = messages.subList(0, Math.min(messages.size(), context.topContext().length()));
                    JsonMap chatData = this.chatsData.get(entry.getKey());
                    if (chatData != null) {
                        chats.put(
                            entry.getKey(),
                            new Chat(chatData, messages));
                    }
                }

                callback.completed(chats);
            } catch (JsonException e) {
                session.logger().log(Level.SEVERE, "JsonParse exception", e);
                failed(e);
            } finally {
                freeJsonParser(jsonParser);
                session.logger().info("Messages callback time: "
                    + (System.currentTimeMillis() - startTime));
            }
        }
    }

    private class MergeCallback
        extends AbstractFilterFutureCallback<List<Map<String, Chat>>, List<Chat>>
    {
        private final TopPostsRequestContext context;

        MergeCallback(
            final FutureCallback<Collection<Chat>> callback,
            final TopPostsRequestContext context)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(
            final List<Map<String, Chat>> shardsResult)
        {
            Map<String, Chat> chatsMap = new LinkedHashMap<>();
            for (Map<String, Chat> shardResult: shardsResult) {
                chatsMap.putAll(shardResult);
            }

            Iterator<Map.Entry<String, Chat>> iterator = chatsMap.entrySet().iterator();
            int total = 0;
            List<Chat> chats = new ArrayList<>(chatsMap.size());

            for (Map.Entry<String, Chat> entry: chatsMap.entrySet()) {
                Chat chat = entry.getValue();
                if (chat.messages().size() <= 0) {
                    context.logger().info("Skipping chat - no messages, " + chat.chatId());
                    iterator.remove();
                } else {
                    context.logger().info("Chat "
                        + chat.chatId() + ", messagesSelected: "
                        + chat.messages().size());
                    total += chat.messages().size();
                }

                chats.add(chat);
            }

            context.logger().info("Total Chats " + chats.size() + " messages " + total);
            callback.completed(chats);
        }
    }

    private static final class JsonParseCallback
        extends AbstractFilterFutureCallback<String, JsonObject>
    {
        private final ThreadPoolExecutor executor;
        private final ProxySession session;
        private final String tag;

        JsonParseCallback(
            final FutureCallback<JsonObject> callback,
            final ThreadPoolExecutor executor,
            final ProxySession session,
            final String tag)
        {
            super(callback);
            this.executor = executor;
            this.session = session;
            this.tag = tag;
        }

        @Override
        public void completed(final String result) {
            session.logger().info("Response received: " + tag);
            executor.execute(new JsonParseTask(result, callback, session));
        }
    }

    private static final class JsonParseTask implements Runnable {
        private final FutureCallback<? super JsonObject> callback;
        private final String msg;
        private final ProxySession session;

        public JsonParseTask(
            final String msg,
            final FutureCallback<? super JsonObject> callback,
            final ProxySession session)
        {
            this.msg = msg;
            this.callback = callback;
            this.session = session;
        }

        @Override
        public void run() {
            SimpleJsonParser parser = getJsonParser();
            long time = System.currentTimeMillis();
            try {
                JsonObject json = parser.parse(msg);
                session.logger().info("Json parse time: "
                    + (System.currentTimeMillis() - time));
                callback.completed(json);
            } catch (Exception e) {
                callback.failed(e);
            } finally {
                freeJsonParser(parser);
            }
        }
    }
}
