package ru.yandex.search.messenger.proxy.topposts;

import java.io.IOException;
import java.util.Collection;
import java.util.logging.Level;

import NMessengerProtocol.Search;
import com.google.protobuf.CodedOutputStream;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.cityhash.CityHashingArrayOutputStream;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

public class TopPostsPrinter extends AbstractProxySessionCallback<Collection<Chat>> {
    private static final ThreadLocal<CityHashingArrayOutputStream> OUT_TLS =
        ThreadLocal.withInitial(
            () -> new CityHashingArrayOutputStream());

    private static final int VERSION = 2;
    private final TopPostsRequestContext context;
    private final JsonType jsonType;

    public TopPostsPrinter(
        final TopPostsRequestContext context,
        final ProxySession session)
        throws BadRequestException
    {
        super(session);

        this.context = context;

        this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
    }

    @Override
    public void completed(final Collection<Chat> chats) {
        try {
            session.logger().info("Chats " + chats.size());
            long startTime = System.currentTimeMillis();
            HttpEntity response;
            if (context.proto()) {
                response = generateProtoResponse(chats);
            } else {
                response = generateJsonResponse(chats);
            }
            session.logger().info("Generate response time: "
                + (System.currentTimeMillis() - startTime));

            session.response(HttpStatus.SC_OK, response);
        } catch (IOException | JsonException e) {
            session.logger().log(Level.SEVERE, "JsonParse exception", e);
            failed(e);
        }
    }

    private HttpEntity generateProtoResponse(
        final Collection<Chat> chats)
        throws IOException
    {
        Search.TTopPostsResponse.Builder builder =
            Search.TTopPostsResponse.newBuilder();
        int messages = 0;
        int[] postCount = new int[1];
        for (Chat chat: chats) {
            messages += chat.messages().size();
            if (chat.messages().size() > 0) {
                builder.addTopPosts(chat.toProto(context.length(), postCount));
            }
        }

        session.logger().info("Messages " + messages);

        CityHashingArrayOutputStream out = OUT_TLS.get();
        out.reset();
        CodedOutputStream googleOut = CodedOutputStream.newInstance(out);
        builder.build().writeTo(googleOut);
        googleOut.flush();

        byte[] postData = out.toByteArrayWithVersion(VERSION);
        return new NByteArrayEntity(postData);
    }

    private HttpEntity generateJsonResponse(
        final Collection<Chat> chats)
        throws IOException, JsonException
    {
        int messages = 0;
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = jsonType.create(sbw)) {
            writer.startArray();
            for (Chat chat: chats) {
                messages += chat.messages().size();
                if (chat.messages().size() > 0) {
                    chat.writeJson(writer, context.length());
                }
            }
            writer.endArray();
        }

        session.logger().info("Messages " + messages);
        return new NStringEntity(
            sbw.toString(),
            ContentType.APPLICATION_JSON.withCharset(
                session.acceptedCharset()));
    }
}
