package ru.yandex.search.messenger.proxy.topposts;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.client.producer.ImmutableProducerClientConfig;
import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.ProducerClientConfigBuilder;
import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.ps.search.messenger.MessageFields;
import ru.yandex.search.messenger.proxy.Moxy;
import ru.yandex.search.messenger.proxy.SimpleJsonParser;
import ru.yandex.search.messenger.proxy.config.ImmutableTopPostsConfig;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class TopPostsUpdater implements Runnable, Stater {
    private final PrefixedLogger updateLogger;
    private final SharedConnectingIOReactor reactor;
    private final ProducerClient producerClient;
    private final AsyncClient backendClient;
    private final Moxy moxy;
    private final int timeout;
    private final SimpleJsonParser jsonParser;
    private final String script;
    private volatile boolean stopped = false;
    private final Map<CgiParams, CachedResponse> cache;
    private final List<CgiParams> staticParams;
    private final Thread thread;
    private final ImmutableTopPostsConfig config;
    private final Set<String> channelsFilter;

    public TopPostsUpdater(final Moxy moxy, final Set<String> channelsFilter) throws IOException {
        config = moxy.config().topPostConfig();
        this.channelsFilter = channelsFilter;
        moxy.registerStater(this);
        thread = new Thread(this, "TopPostsThread");
        thread.setDaemon(true);
        reactor = new SharedConnectingIOReactor(moxy.config(), moxy.config().dnsConfig());
        reactor.start();
        ImmutableProducerClientConfig producerConfig;
        if (moxy.config().producerClientConfig() != null) {
            try {
                producerConfig = new ProducerClientConfigBuilder()
                    .streaming(false)
                    .connections(2)
                    .host(moxy.config().producerClientConfig().host())
                    .build();
                producerClient =
                    new ProducerClient(reactor, producerConfig, moxy.searchMap());
                producerClient.start();
            } catch (ConfigException ce) {
                throw new IOException(ce);
            }
        } else {
            producerClient = null;
        }

        staticParams = new ArrayList<>();
        for (String paramsStr: config.cachedParams()) {
            staticParams.add(new CgiParams(paramsStr));
        }

        this.moxy = moxy;
        this.timeout = (int) TimeUnit.SECONDS.toMillis(60);
        this.jsonParser = new SimpleJsonParser();
        this.updateLogger = moxy.logger().addPrefix("TopPostsUpdater");

        this.script = IOStreamUtils.consume(
            new InputStreamReader(
                Moxy.class.getResourceAsStream("top_posts_request.js"),
                StandardCharsets.UTF_8))
            .toString();

        backendClient = new AsyncClient(reactor, moxy.config().searchConfig());
        backendClient.start();

        this.cache = new ConcurrentHashMap<>();

        int maxAttempt = 1;
        if (config.startupAfterNumberFailures() > 0) {
            maxAttempt = config.startupAfterNumberFailures();
        }

        int attempt = 0;
        Exception error = null;
        while (attempt < maxAttempt) {
            try {
                for (CgiParams params: staticParams) {
                    process(params);
                }

                error = null;
                break;
            } catch (Exception e) {
                error = e;
                updateLogger.log(Level.WARNING, "Exception on loading top posts", e);

                try {
                    Thread.sleep(5000);
                } catch (InterruptedException ie) {
                    throw new IOException(ie);
                }
            }

            attempt += 1;
        }

        if (error != null) {
            if (config.startupAfterNumberFailures() <= 0) {
                throw new RuntimeException("Failed to load top posts", error);
            } else {
                updateLogger.log(Level.WARNING, "Keep starting after failures on top posts", error);
            }
        }

        thread.start();
    }

    @Override
    public <E extends Exception> void stats(StatsConsumer<? extends E> statsConsumer) throws E {
//        statsConsumer.stat(
//            "top-posts-chats-count_axxx",
//            cache.get().chats().size());
//        statsConsumer.stat(
//            "top-posts-chats-count_annn",
//            cache.get().chats().size());
    }

    @Override
    public void run() {
        try {
            while (!stopped) {
                long paramsInterval = config.updateInterval() / staticParams.size();
                for (CgiParams params: staticParams) {
                    try {
                        process(params);
                    } catch (Exception e) {
                        updateLogger.log(
                            Level.WARNING,
                            "Failed to update channels",
                            e);
                    }

                    Thread.sleep(paramsInterval);
                }
            }
        } catch (InterruptedException ie) {
            updateLogger.log(
                Level.WARNING,
                "Update thread interrupted",
                ie);
        }
    }

    public CachedResponse cache(final CgiParams params) {
        return cache.get(params);
    }

    protected void process(final CgiParams params) throws Exception {
        CachingCallback callback = new CachingCallback(params);

        TopPostsRequestContext topContext =
            new TopPostsRequestContext(params, updateLogger, moxy.config());
        ChatsRequestContext chatsContext = new ChatsRequestContext(topContext);

        List<QueueHostInfo> info =
            producerClient.executeWithInfo(chatsContext.user()).get(timeout, TimeUnit.MILLISECONDS);

        List<HttpHost> hosts = info.stream().map((x) -> x.host()).collect(Collectors.toList());
        String query = chatsContext.query().toString();
        JsonObject chatsResponse = backendClient.execute(
            hosts,
            new BasicAsyncRequestProducerGenerator(query),
            System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE).get();
        processChatsResponse(topContext, callback, chatsResponse.asMap().getList("hitsArray"));
    }

    protected void processChatsResponse(
        final TopPostsRequestContext context,
        final FutureCallback<Map<String, Chat>> callback,
        final JsonList hits) {
        try {
            Map<String, JsonMap> chats = new LinkedHashMap<>();
            for (JsonObject o: hits) {
                JsonMap hit = o.asMap();
                String chatId = hit.getString("chat_id");
                if (channelsFilter != null && !channelsFilter.contains(chatId)) {
                    context.logger().info("Channel filtered by hardcoded list");
                    continue;
                }
                chats.put(chatId, hit);
            }
            if (chats.size() == 0) {
                String msg =
                    "No chats found with show_on_morda flag";
                context.logger().severe(msg);

                callback.completed(Collections.emptyMap());
                return;
            } else {
                processChatsMessages(
                    context,
                    callback,
                    chats);
            }
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    protected void processChatsMessages(
        final TopPostsRequestContext context,
        final FutureCallback<Map<String, Chat>> callback,
        final Map<String, JsonMap> chats)
        throws Exception
    {
        Map<SearchMapShard, Map<String, JsonMap>> shardMap = new HashMap<>();
        Map<SearchMapShard, User> userMap = new HashMap<>();
        for (Map.Entry<String, JsonMap> chat: chats.entrySet()) {
            String chatId = chat.getKey();
            Prefix prefix = new StringPrefix(chatId);
            User user = new User(moxy.config().messagesService(), prefix);
            SearchMapShard shard = moxy.searchMap().apply(user);
            Map<String, JsonMap> shardChats = shardMap.get(shard);
            if (shardChats == null) {
                shardChats = new LinkedHashMap<>(chats.size());
                shardMap.put(shard, shardChats);
                userMap.put(shard, user);
            }
            shardChats.put(chatId, chat.getValue());
        }


        Map<String, Chat> result = new LinkedHashMap<>();
        for (Map.Entry<SearchMapShard, Map<String, JsonMap>> entry: shardMap.entrySet()) {
            if (entry.getValue().isEmpty()) {
                continue;
            }
            ShardMessagesRequestContext shardContext =
                new ShardMessagesRequestContext(context, false, entry.getValue());
            result.putAll(getChatMessages(shardContext));
        }

        callback.completed(result);
    }

    protected Map<String, Chat> getChatMessages(
        final ShardMessagesRequestContext context)
        throws Exception
    {
        if (context.chatIds().isEmpty()) {
            return Collections.emptyMap();
        }

        List<QueueHostInfo> info =
            producerClient.executeWithInfo(context.user()).get(timeout, TimeUnit.MILLISECONDS);
        List<HttpHost> hosts = info.stream().map((x) -> x.host()).collect(Collectors.toList());

        String prefixes = StringUtils.join(context.chatIds(), "\",\"", "[\"", "\"]");

        String shardScript =
            script.replace("%PREFIXES%", prefixes).replace("%SEARCH_URI%", context.createQuery().toString());

        context.logger().info("Script " + shardScript);
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/execute?top_posts_update",
                new StringEntity(shardScript, StandardCharsets.UTF_8));

        JsonObject messagesResponse = backendClient.execute(
            hosts,
            generator,
            System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE).get();

        context.logger().info(
            "Messages response: "
                + JsonType.HUMAN_READABLE.toString(messagesResponse));
        Map<String, Chat> chats = new LinkedHashMap<>(context.chatIds().size());

        JsonList hits = messagesResponse.asList();

        for (JsonObject itemObj : hits) {
            JsonMap item = itemObj.asMap();
            String chatId = item.getString("prefix");
            JsonList messagesList = item.getList("messages");
            List<Message> messages = new ArrayList<>(messagesList.size());
            for (JsonObject messageObj: messagesList) {
                JsonMap hit = messageObj.asMap();
                String chatIdStr = hit.getOrNull(MessageFields.CHAT_ID.stored());
                if (chatIdStr == null) {
                    continue;
                }
                messages.add(new Message(messageObj.asMap(), jsonParser));
            }

            context.logger().info("For chat " + chatId + " messages count " + messages.size());
            JsonMap chatJson = context.chats().get(chatId);
            if (chatJson == null) {
                context.logger().warning("For chat " + chatId + " no chat info");
                continue;
            }

            Chat chat = new Chat(chatJson, messages);
            chats.put(chatId, chat);
        }

        return chats;
    }

    private final class CachingCallback
        implements FutureCallback<Map<String, Chat>>
    {
        private final long startTs = System.currentTimeMillis();
        private final CgiParams params;

        public CachingCallback(final CgiParams params) {
            this.params = params;
        }

        @Override
        public void completed(final Map<String, Chat> chats) {
            long ts = System.currentTimeMillis();
            long elapsed = TimeUnit.MILLISECONDS.toSeconds(ts - startTs);
            cache.put(params, new CachedResponse(chats, System.currentTimeMillis()));
            updateLogger.info("Top posts successfully updated in " + elapsed + " seconds");
        }

        @Override
        public void failed(Exception e) {
            long ts = System.currentTimeMillis();
            long elapsed = TimeUnit.MILLISECONDS.toSeconds(ts - startTs);
            updateLogger.info("Top posts update failed in " + elapsed + " seconds");
        }

        @Override
        public void cancelled() {
            long ts = System.currentTimeMillis();
            long elapsed = TimeUnit.MILLISECONDS.toSeconds(ts - startTs);
            updateLogger.info("Top posts update cancelled in " + elapsed + " seconds");
        }
    }

    public static class CachedResponse {
        private final long ts;
        private final Map<String, Chat> chats;

        public CachedResponse(
            final Map<String, Chat> chats,
            final long ts)
        {
            this.ts = ts;
            this.chats = chats;
        }

        public long ts() {
            return ts;
        }

        public Map<String, Chat> chats() {
            return chats;
        }
    }
}
