#!/usr/bin/python
# -*- coding: utf-8 -*-
import tempfile
import threading
import errno

from optparse import OptionParser

from mail_search_mock_utils import *


class Queue(MailSearchService):
    def __init__(self, name, port, zks):
        MailSearchService.__init__(self, 'mail_search_queue_prod4', port, name=name)
        self.zks = zks

    def prepare(self):
        config = parse_config(self.path('mail-search-queue-prod.cfg.template'))
        for key in config.keys():
            if key.startswith('server.') or key.startswith('weight.') or key.startswith('group.'):
                del config[key]

        config['clientPort'] = str(self.port)
        config['httpPort'] = str(self.port + 1)
        for i in range(len(self.zks)):
            config['server.' + str(i + 1)] = zks[i][0] + ':' + str(self.zks[i][1] + 2) + ':' + str(self.zks[i][1] + 3)

        config['group.1'] = '1:2:3:4'

        config['weight.1'] = '1'
        config['weight.2'] = '1'
        config['weight.3'] = '1'
        config['weight.4'] = '2'
        print_config(config, self.path('mail-search-queue-prod.cfg.template'))

    def envs(self):
        return {
            'BSCONFIG_IHOST': self.name,
            'BSCONFIG_IPORT': str(self.port),
            'BSCONFIG_ITAGS': '""',
            'BSCONFIG_INAME': self.name + ':' + str(self.port),
            'BSCONFIG_IDIR': str(self.dir)}


class PerseusService(MailSearchService):
    def __init__(self, port, searchmap, hostmap, producer_port=17000):
        MailSearchService.__init__(self, 'mail_search_prod', port, 'perseus-' + str(port))
        self.producer_port = producer_port
        self.hostmap = hostmap
        self.searchmap = searchmap
        self.extract = True

    def prepare(self):
        with open(self.path('check_overrides.disable'), 'w'):
            pass

        with open(self.path('copier.sh'), 'w') as copier:
            copier.write('#!/bin/bash\n')
            copier.write('exit 0')

        lucene_config = parse_config(self.path('thin_lucene.conf'))
        lucene_config['check-copyness'] = 'false'
        print_config(lucene_config, self.path('thin_lucene.conf'))

        consumer_config = parse_config(self.path('consumer.conf'))
        consumer_config['zoolooser.producers'] = 'localhost:' + str(self.producer_port)
        consumer_config['system_cmd_resolver.cmd'] = 'echo localhost'
        print_config(consumer_config, self.path('consumer.conf'))

        kamaji_config = parse_config(self.path('kamaji.conf'))
        replace_hosts_in_config(kamaji_config, self.hostmap)
        kamaji_config['tvm.client-id'] = TVM_CLIENT_ID
        kamaji_config['tvm.secret'] = SECRET
        kamaji_config['tvm.robot-uid'] = ROBOT_UID
        kamaji_config['tvm.robot-key'] = ROBOT_KEY
        kamaji_config['blackbox.tvm2.client-id'] = TVM_CLIENT_ID
        kamaji_config['blackbox.tvm2.destination-client-id'] = BLACKBOX_CLIENT_ID
        kamaji_config['blackbox.tvm2.secret'] = SECRET
        print_config(kamaji_config, self.path('kamaji.conf'))

        with open(self.path('searchmap.txt'), 'w') as sm:
            sm.write(self.searchmap)

        ljinx_conf_home = self.path('ljinx-bundle')
        os.mkdir(ljinx_conf_home)
        shutil.copy(self.path('perseus-bundle/mail-search-backend-ljinx.conf'), ljinx_conf_home)
        replace_paths_in_file(os.path.join(ljinx_conf_home, 'mail-search-backend-ljinx.conf'), self.dir)
        shutil.copy(self.path('perseus-bundle/ljinx.conf'), ljinx_conf_home)
        replace_paths_in_file(os.path.join(ljinx_conf_home, 'ljinx.conf'), self.dir)

        shutil.copytree(self.path('perseus-bundle/conf.d'), os.path.join(ljinx_conf_home, 'conf.d'))
        conf_d = os.path.join(ljinx_conf_home, 'conf.d')
        for config in os.listdir(conf_d):
            config_path = os.path.join(conf_d, config)
            if os.path.isfile(config_path):
                replace_paths_in_file(config_path, self.dir)

        shutil.copytree(self.path('perseus-bundle/common-conf.d'), os.path.join(ljinx_conf_home, 'common-conf.d'))
        common_conf_d = os.path.join(ljinx_conf_home, 'common-conf.d')
        for config in os.listdir(common_conf_d):
            config_path = os.path.join(common_conf_d, config)
            if os.path.isfile(config_path):
                replace_paths_in_file(config_path, self.dir)

        if sp.call(['sed', '-i', 's/perseus-bundle\/mail-search-backend-ljinx.conf/ljinx-bundle\/mail-search-backend-ljinx.conf/g', self.path('instancectl.conf')]) != 0:
            print 'Failed replace config in instancectl'
            exit(1)


class MailSaloProducer(MailSearchService):
    def __init__(self, searchmap, port):
        MailSearchService.__init__(self, 'mail_salo_producer', port)
        self.searchmap = searchmap

    def prepare(self):
        with open(self.path('salo-searchmap.txt'), 'w') as smp:
            smp.write(self.searchmap)

class MailSalo(MailSearchService):
    def __init__(self, port, producer_port, msal_port, units):
        MailSearchService.__init__(self, 'mail_salo', port)
        self.hostmap = {
            'http://$(PORTO_HOST):8081/': 'localhost:' + str(producer_port),
            'http://msal.mail.yandex.net:8080/': 'localhost:' + str(msal_port)
        }
        self.units = units

    def envs(self):
        envs = MailSearchService.envs(self)
        envs['PORTO_HOST'] = 'localhost'
        return envs

    def prepare(self):
        if sp.call(['sed', '-i', 's/units-search.mail.yandex.net/' + self.units + '/g', self.path('instancectl.conf')]) != 0:
            print 'Failed to modify units server to ', self.units, ' in instancectl.conf for', self.name
            exit(1)

        salo_conf = parse_config(self.path('salo.conf'))
        replace_hosts_in_config(salo_conf, self.hostmap)
        print_config(salo_conf, self.path('salo.conf'))


class Msal(MailSearchService):
    def __init__(self, port, blackbox, sharpei):
        MailSearchService.__init__(self, 'msal', port)
        self.blackbox = blackbox
        self.sharpei = sharpei

    def gather_resources(self):
        pass

    def load(self):
        os.symlink(os.path.abspath('msal.conf'), self.path('msal.conf'))
        extract(sandbox_download('164374429', self.path('ibm-java-jre-1.8.tar.gz')), self.dir)
        extract(sandbox_download('461254296', self.path('msal.tar.gz')), self.dir)

    def start_service(self):
        envs = self.envs()
        envs['BLACKBOX_CORP'] = self.blackbox
        envs['BLACKBOX'] = self.blackbox
        envs['SHARPEI'] = self.sharpei

        cwd = os.getcwd()
        os.chdir(self.dir)
        args = [
            self.path('usr/lib/jvm/yandex-ibm-java/jre/bin/java'),
            '-Xmr16M',
            '-Xconcurrentbackground6',
            '-Xconcurrentlevel8',
            '-Xgcthreads8',
            '-Xms1024M',
            '-Xmx1024M',
            '-Xpartialcompactgc',
            '-Djava.net.preferIPv4Stack=false',
            '-Djava.net.preferIPv6Addresses=true',
            '-cp', self.path('msal-bundle/msal.jar'),
            'ru.yandex.search.msal.Server',
            self.path('msal.conf')]

        print 'Msal args', args
        self.service_process = sp.Popen(args, env=envs)
        os.chdir(cwd)
        print 'Msal started', self.service_process.pid, 'port', self.port


class Tikaite(MailSearchService):
    def __init__(self, port, lenulka):
        MailSearchService.__init__(self, 'mail_search_tupita', port)
        self.lenulka = lenulka

    def prepare(self):
        os.remove(self.path('instancectl.conf'))
        os.symlink(os.path.abspath('tikaite_instancectl.conf'), self.path('instancectl.conf'))

        with open(self.path('tikaite.conf'), 'w') as tikaite_config:
            tikaite_config.write(
                "log.level.min = all\nserver.port = " + str(self.port) + "\nserver.workers.min = 2\n"
                + "accesslog.file=" + self.path("tikaite-access.log") + '\n'
                + "log.file=" + self.path("tikaite-full.log") + '\n'
                + "stderr.file=" + self.path("tikaite-error.log") + '\n'
                + "stdout.file=" + self.path("tikaite-out.log") + '\n'
                + "server.workers.percent = 50\nserver.connections = 50\n"
                + "server.timeout = 5000\nstorage.timeout = 5000\n"
                + "storage.connections = 100\nstorage.host = " + str(self.lenulka[0])
                + "\nstorage.port = " + str(self.lenulka[1]) + '\n')


class MsearchProxy(MailSearchService):
    def __init__(self, port, searchmap, hostmap):
        MailSearchService.__init__(self, 'mail_msearch_proxy', port)
        self.hostmap = hostmap
        self.searchmap = searchmap

    def prepare(self):
        manual = self.path('searchmap_manual.txt')
        if os.path.exists(manual):
            os.remove(manual)

        proxy_config_name = 'msearch-proxy.conf'
        proxy_config = parse_config(self.path(proxy_config_name))
        proxy_config['tvm.client-id'] = TVM_CLIENT_ID
        proxy_config['tvm.secret'] = SECRET
        proxy_config['tvm.robot-uid'] = ROBOT_UID
        proxy_config['tvm.robot-key'] = ROBOT_KEY
        proxy_config['blackbox.tvm2.client-id'] = TVM_CLIENT_ID
        proxy_config['blackbox.tvm2.destination-client-id'] = BLACKBOX_CLIENT_ID
        proxy_config['blackbox.tvm2.secret'] = SECRET
        proxy_config['corp-blackbox.tvm2.client-id'] = TVM_CLIENT_ID
        proxy_config['corp-blackbox.tvm2.destination-client-id'] = BLACKBOX_CLIENT_ID
        proxy_config['corp-blackbox.tvm2.secret'] = SECRET
        replace_hosts_in_config(proxy_config, self.hostmap)
        print_config(proxy_config, self.path(proxy_config_name))

        with open(self.path('searchmap.txt'), 'w') as sm:
            sm.write(self.searchmap)

def searchmap(ports, zks):
    return '\n'.join([searchmap_item(port, zks) for port in ports])

def searchmap_item(port, zks):
    line = 'change_log iNum:0,tag:localhost_' + str(port)
    line += ',host:localhost,shards:0-65533,zk:'
    zk_line = [zk[0] + ':' + str(zk[1]) + '/' + str(zk[1] + 1) for zk in zks]
    line += '|'.join(zk_line)
    line += ',json_indexer_port:' + str(port + 4)
    line += ',search_port_ng:' + str(port + 1)
    line += ',search_port:' + str(port)
    return line


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def adjust_http(host):
    if not unicode(host).startswith('http://'):
        return 'http://' + host


if __name__ == '__main__':
    parser = OptionParser()
    parser.add_option("-p", "--bind-port", dest="bindport", default=14550, help="Start port allowed to bind", type=int)
    parser.add_option("-w", "--work-dir", dest="workdir", help="working directory", default=os.getcwd())
    parser.add_option("-l", "--lenulka", dest="storage", help="Lenulka host:port")
    parser.add_option("-b", "--blackbox", dest="blackbox", help="Blackbox host:port")
    parser.add_option("-f", "--filter-search", dest="filter_search", help="Filter Search host:port")
    parser.add_option("-s", "--sharpei", dest="sharpei", help="Sharpei host:port")
    parser.add_option("-u", "--units", dest="units", help="units.mail.yandex.net host:port")
    parser.add_option("-n", "--nanny-key", dest="nannykey", default=None, help="Nanny oauth key")

    (options, args) = parser.parse_args()

    os.chdir(os.path.abspath(options.workdir))

    lenulka = parse_host(options.storage, parser)

    parse_host(options.units, parser)
    parse_host(options.sharpei, parser)
    parse_host(options.filter_search, parser)
    parse_host(options.blackbox, parser)

    if 'NANNY_OAUTH_KEY' not in os.environ:
        if options.nannykey is not None:
            os.environ['NANNY_OAUTH_KEY'] = str(options.nannykey)
        else:
            print 'Missing NANNY_OAUTH_KEY eviroment variable'
            sys.exit(1)

    print 'Proxy port is', options.bindport

    start_port = int(options.bindport)
    proxy_port = options.bindport
    lucene_port_1 = options.bindport + 10
    lucene_port_2 = options.bindport + 2 * 10
    producer_port = options.bindport + 3 * 10
    tvm_port = options.bindport + 4 * 10
    tikaite_port = options.bindport + 5 * 10
    msal_port = options.bindport + 6 * 10
    salo_port = options.bindport + 7 * 10
    queue1_port = options.bindport + 8 * 10
    queue2_port = options.bindport + 9 * 10
    queue3_port = options.bindport + 10 * 10
    queue4_port = options.bindport + 11 * 10

    print 'Engaged ports are from', options.bindport, 'to', queue4_port

    hostmap = {
        'http://blackbox.yandex.net': adjust_http(options.blackbox),
        'http://blackbox-ipv6.yandex.net': adjust_http(options.blackbox),
        'http://blackbox-ipv6.yandex-team.ru': adjust_http(options.blackbox),
        'http://webattach.mail.yandex.net/message_part_real/': adjust_http(options.filter_search) + 'message_part_real/',
        'http://meta.mail.yandex.net:9090/filter_search?caller=msearch': adjust_http(options.filter_search) + '/filter_search?caller=msearch',
        'http://meta.mail.yandex.net:9090/filter_search': adjust_http(options.filter_search) + '/filter_search',
        'http://meta.mail.yandex.net:9090/threads_info?': adjust_http(options.filter_search) + '/threads_info?',
        'http://meta.mail.yandex.net:9090/labels': adjust_http(options.filter_search) + '/labels',
        'http://meta.mail.yandex.net:9090/folders': adjust_http(options.filter_search) + '/folders',
        'http://metacorp.mail.yandex.net:9090/filter_search': adjust_http(options.filter_search) + '/filter_search',
        'http://metacorp.mail.yandex.net:9090/labels': adjust_http(options.filter_search) + '/labels',
        'http://metacorp.mail.yandex.net:9090/folders': adjust_http(options.filter_search) + '/folders',
        '$(TVM_API_HOST)': 'http://localhost:' + str(tvm_port),
        '$(TIKAITE_SRW_HOST)': 'http://localhost:' + str(tikaite_port),
        'http://uaas.search.yandex.net:80': 'http://localhost:' + str(tvm_port),
        'http://salo-producer.n.yandex-team.ru:80': 'http://localhost:' + str(producer_port),
        'http://msal.mail.yandex.net:8080': 'http://localhost:' + str(msal_port),
        'http://sosearch.so.yandex.net/check': 'http://localhost:' + str(tvm_port),
        'http://misc-spell.yandex.net:19036': 'http://localhost:' + str(tvm_port),
        'iex-proxy.search.yandex.net:21953': 'http://localhost:' + str(tvm_port)
    }

    services = []
    zks = [
        ('mail-search-queue1', queue1_port),
        ('mail-search-queue2', queue2_port),
        ('mail-search-queue3', queue3_port),
        ('mail-search-queue4', queue4_port)]

    #hosts_file = tempfile.NamedTemporaryFile('w', suffix='_mail_search.hosts')
    hosts_content = ''
    if os.path.exists('/etc/hosts'):
        with open('/etc/hosts', 'r') as hosts_file:
            hosts_content = hosts_file.read()

    if hosts_content.find('mail-search-queue') < 0:
        hosts_content += '\n'
        hosts_content += '127.0.0.1 mail-search-queue1\n'
        hosts_content += '127.0.0.1 mail-search-queue2\n'
        hosts_content += '127.0.0.1 mail-search-queue3\n'
        hosts_content += '127.0.0.1 mail-search-queue4\n'
        with open('/etc/hosts', 'w') as hosts_file:
            hosts_file.write(hosts_content)

    smp = searchmap([lucene_port_1, lucene_port_2], zks)
    os.environ['HOSTALIASES'] = hosts_file.name

    msal = Msal(msal_port, options.blackbox, options.sharpei)
    services.append(msal)
    tvm_server = TvmStaticServer(("", tvm_port), TvmStaticHandler)

    queue1 = Queue(zks[0][0], zks[0][1], zks)
    queue2 = Queue(zks[1][0], zks[1][1], zks)
    queue3 = Queue(zks[2][0], zks[2][1], zks)
    queue4 = Queue(zks[3][0], zks[3][1], zks)
    services.append(queue1)
    services.append(queue2)
    services.append(queue3)
    services.append(queue4)
    #
    salo_producer = MailSaloProducer(smp, producer_port)
    services.append(salo_producer)

    salo = MailSalo(salo_port, producer_port, msal_port, options.units)
    services.append(salo)

    tikaite = Tikaite(tikaite_port, lenulka)
    services.append(tikaite)
    perseus1 = PerseusService(lucene_port_1, smp, hostmap, producer_port)
    services.append(perseus1)
    proxy = MsearchProxy(proxy_port, smp, hostmap)
    services.append(proxy)
    perseus2 = PerseusService(lucene_port_2, smp, hostmap, producer_port)
    services.append(perseus2)

    sandbox_rids = {}

    sandbox_cache_path = os.getcwd() + '/sandbox-cache/'
    if os.path.exists(sandbox_cache_path):
        shutil.rmtree(sandbox_cache_path)

    os.mkdir(sandbox_cache_path)
    for service in services:
        for sf in service.resources['sandbox']:
            sandbox_rids[str(sf.resource_id).strip()] = sandbox_cache_path + str(sf.resource_id) + '-' + sf.local_path

    # download
    print 'Downloading sandbox resources', sandbox_rids
    download_threads = []
    for rid, name in sandbox_rids.iteritems():
        thread = threading.Thread(target=sandbox_download, args=(rid, name))
        download_threads.append(thread)
        thread.start()

    for thread in download_threads:
        thread.join()

    print 'Resources fetched, preparing services'
    # prepare
    prepare_threads = []
    for service in services:
        thread = threading.Thread(target=lambda s: s.init_service(), args=(service, ))
        prepare_threads.append(thread)
        thread.start()

    for thread in prepare_threads:
        thread.join()

    print 'All prepared, starting'
    signal_handler = TerminationHandler(services)

    for service in services:
        print 'Starting', service.name
        service.start_service()

    tvm_server.serve_forever()
    signal_handler.handler()
