package ru.yandex.search.mop.manager;

import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.Timer;
import java.util.logging.Level;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mop.common.parsers.SearchMapJsonParser;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.searchmap.SearchMapContainer;
import ru.yandex.search.mop.manager.config.ImmutableMopManagerConfig;

public class MopManager {
    private final SearchMapContainer searchMapContainer;
    private final Timer timer;
    private final UpdateSearchMapTask task;
    private final ImmutableMopManagerConfig config;
    private final PrefixedLogger logger;
    private SearchMapWrapper searchMapWrapper;

    public MopManager(
        final ImmutableMopManagerConfig config,
        final ImmutableDnsConfig dnsConfig,
        final PrefixedLogger logger)
    {
        this.searchMapContainer = new SearchMapContainer();
        this.config = config;
        this.logger = logger;
        this.timer = new Timer("MopTimer", true);
        this.task = new UpdateSearchMapTask(this, dnsConfig, logger);
    }

    public void updateSearchMap(final SearchMap searchMap)
        throws Exception
    {
        searchMapContainer.searchMap(searchMap);
        searchMapWrapper = new SearchMapWrapper(
            searchMap,
            config.searchPortName(),
            config.indexPortName());
    }

    public SearchMapContainer searchMapContainer() {
        return searchMapContainer;
    }

    public SearchMapWrapper searchMapWrapper() {
        if (searchMapWrapper == null) {
            uploadSearchMap();
        }
        return searchMapWrapper;
    }

    public ImmutableMopManagerConfig config() {
        return config;
    }

    public void start() {
        if (!config.pumpkin()) {
            timer.schedule(task, 0, config.interval());
        }
    }

    private void uploadSearchMap() {
        if (config.pumpkin()) {
            uploadSearchMapFromFile();
        } else {
            try {
                uploadSearchMapFromMop();
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Searchmap uploading from mop failed", e);
                uploadSearchMapFromFile();
            }
        }
    }

    private void uploadSearchMapFromMop() throws Exception {
        SearchMap searchMap = task.requestSearchMap();
        updateSearchMap(searchMap);
        logger.info("Searchmap uploaded from mop, version "
            + searchMap.version());
        task.writeSearchMapToFile();
        logger.info("Searchmap was written to file");
    }

    private void uploadSearchMapFromFile() {
        try (Reader reader = new InputStreamReader(
            new FileInputStream(config.file()),
            StandardCharsets.UTF_8))
        {
            SearchMap searchMap = SearchMapJsonParser.INSTANCE.parse(reader);
            updateSearchMap(searchMap);
            logger.info("Searchmap uploaded from file, version "
                + searchMap.version());
        } catch (Exception e) {
            throw new RuntimeException("Searchmap uploading from file failed", e);
        }
    }
}
