package ru.yandex.search.mop.manager;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpHost;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapConfigBuilder;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.ZooKeeperAddress;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.prefix.PrefixType;

public class SearchMapWrapper extends SearchMap {
    private static final String SEARCH_PORT = "search_port";
    private static final String QUEUE_ID_PORT = "queue_id_port";

    public SearchMapWrapper(
        final ru.yandex.search.mop.common.searchmap.SearchMap searchMap,
        final String searchPortName,
        final String indexPortName)
        throws IOException, ParseException, ConfigException
    {
        super(new SearchMapConfigBuilder().content("").build());
        map = convertSearchMap(searchMap, searchPortName, indexPortName);
    }

    private Map<String, SearchMapRow> convertSearchMap(
        final ru.yandex.search.mop.common.searchmap.SearchMap searchMap,
        final String searchPortName,
        final String indexPortName)
    {
        Map<String, SearchMapRow> map = new HashMap<>();

        List<List<ZooKeeperAddress>> zkCache =
            new ArrayList<>(searchMap.queues().size());
        for (Queue queue: searchMap.queues()) {
            List<ZooKeeperAddress> list = new ArrayList<>(queue.hosts().size());
            for (QueueHost host: queue.hosts()) {
                list.add(convertQueueHost(host));
            }
            zkCache.add(list);
        }

        List<List<SearchMapHost>> backendHostsCache =
            new ArrayList<>(searchMap.hostGroups().size());
        for (HostGroup group: searchMap.hostGroups()) {
            Set<BackendHost> hosts = group.hosts();
            List<SearchMapHost> list = new ArrayList<>(hosts.size());
            for (BackendHost host: hosts) {
                list.add(convertBackendHost(host, searchPortName, indexPortName));
            }
            backendHostsCache.add(list);
        }

        for (Metashard metashard: searchMap.metashards()) {
            SearchMapRow row = map.get(metashard.service().lowName());
            if (row == null) {
                row = new SearchMapRow(PrefixType.LONG);
                map.put(metashard.service().lowName(), row);
            }
            int start = metashard.shardsRange().start();
            int end = metashard.shardsRange().end();
            int label = metashard.label();
            int queueId = metashard.queueId();
            int hostGroupId = metashard.hostGroupId();
            SearchMapShard shard = new SearchMapShard(zkCache.get(queueId), label);
            for (SearchMapHost host: backendHostsCache.get(hostGroupId)) {
                shard.append(host);
            }
            for (int i = start; i <= end; ++i) {
                row.replace(i, shard);
            }
        }
        Map<SearchMapShard, SearchMapShard> shardsCache = new HashMap<>();
        Map<SearchMapRow, SearchMapRow> rowsCache = new HashMap<>();
        for (Map.Entry<String, SearchMapRow> entry: map.entrySet()) {
            SearchMapRow row = entry.getValue();
            for (int i = 0; i < SearchMap.SHARDS_COUNT; ++i) {
                SearchMapShard shard = row.get(i);
                if (shard == null) {
                    row.replace(i, SearchMap.EMPTY_SHARD);
                } else {
                    SearchMapShard cached = shardsCache.get(shard);
                    if (cached == null) {
                        shardsCache.put(shard, shard);
                    } else {
                        row.replace(i, cached);
                    }
                }
            }
            SearchMapRow cached = rowsCache.get(row);
            if (cached == null) {
                rowsCache.put(row, row);
            } else {
                entry.setValue(cached);
            }
        }
        return map;
    }

    private ZooKeeperAddress convertQueueHost(final QueueHost host) {
        return new ZooKeeperAddress(
            new HttpHost(host.hostname(), host.httpPort()),
            host.zkPort());
    }

    private SearchMapHost convertBackendHost(
        final BackendHost host,
        final String searchPortName,
        final String indexPortName)
    {
        int searchPort;
        int indexPort;
        if (searchPortName.equals(SEARCH_PORT)) {
            searchPort = host.searchPort();
        } else {
            searchPort = host.searchPortNg();
        }
        if (indexPortName.equals(QUEUE_ID_PORT)) {
            indexPort = host.queueIdPort();
        } else {
            indexPort = host.indexPort();
        }
        return new SearchMapHost(host.hostname(), searchPort, indexPort);
    }
}
