package ru.yandex.search.mop.manager;

import java.io.FileWriter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.TimerTask;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mop.common.parsers.SearchMapJsonParser;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.writers.SearchMapJsonWriter;
import ru.yandex.search.mop.manager.config.ImmutableMopManagerConfig;

public class UpdateSearchMapTask extends TimerTask {
    private final MopManager mopManager;
    private final PrefixedLogger logger;
    private final CloseableHttpClient client;
    private final ImmutableMopManagerConfig config;

    public UpdateSearchMapTask(
        final MopManager mopManager,
        final ImmutableDnsConfig dnsConfig,
        final PrefixedLogger logger)
    {
        this.mopManager = mopManager;
        this.config = mopManager.config();
        this.logger = logger;
        this.client = ClientBuilder.createClient(config, dnsConfig);
    }

    @Override
    public void run() {
        try {
            SearchMap newSearchMap = requestSearchMap();
            if (newSearchMap != null) {
                mopManager.updateSearchMap(newSearchMap);
                logger.info("Update searchmap task done, version " +
                    newSearchMap.version());
                writeSearchMapToFile();
                logger.info("Searchmap was written to file");
            } else {
                logger.info("Searchmap is actual");
            }
        } catch (Throwable t) {
            logger.log(Level.SEVERE, "Update searchmap task error", t);
        }
    }

    public SearchMap requestSearchMap()
        throws HttpException, IOException, JsonException, URISyntaxException
    {
        URI uri = getSearchMapURI();
        logger.info("Requesting searchmap");
        HttpGet get = new HttpGet(uri);
        try (CloseableHttpResponse response = client.execute(get)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(uri.toString(), response);
            }
            String body = CharsetUtils.toString(response.getEntity());
            return SearchMapJsonParser.INSTANCE.parse(body);
        } catch (Throwable t) {
            logger.warning(
                "Failed to receive searchmap for request " + uri.toString());
            throw t;
        }
    }

    private URI getSearchMapURI() throws URISyntaxException {
        StringBuilder query = new StringBuilder("&format=json");
        if (mopManager.searchMapContainer().searchMap() != null) {
            long version = mopManager.searchMapContainer().searchMap().version();
            query.append("&version=");
            query.append(version);
        }

        for (String scope: config.scopes()) {
            query.append("&scope=");
            query.append(scope);
        }

        return new URI(
            config.host().getSchemeName(),
            null,
            config.host().getHostName(),
            config.host().getPort(),
            "/get-searchmap",
            query.toString(),
            null)
            .parseServerAuthority();
    }

    public void writeSearchMapToFile() throws IOException, NotFoundException {
        try (FileWriter file = new FileWriter(config.file())) {
            StringBuilderWriter sbw = SearchMapJsonWriter.INSTANCE.write(
                mopManager.searchMapContainer().searchMap(),
                JsonType.NORMAL);
            file.write(sbw.toString());
            file.flush();
        }
    }
}
