package ru.yandex.search.mop.common.parsers;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.searchmap.ShardsRange;
import ru.yandex.search.mop.common.services.Service;

public class SearchMapDBParser {
    public static final SearchMapDBParser INSTANCE = new SearchMapDBParser();

    private static final int SHARDS_COUNT = 65534 << 2;

    private static final String QUEUE_QUERY = "SELECT * FROM queue";
    private static final String HOST_GROUP_QUERY = "SELECT * FROM host_group";
    private static final String METASHARD_QUERY = "SELECT * FROM metashard";

    private SearchMapDBParser() {
    }

    public SearchMap parse(final Connection connection) throws SQLException {
        List<Queue> queues = uploadQueues(connection);
        List<HostGroup> hostGroups = uploadHostGroups(connection);
        return uploadSearchMap(connection, queues, hostGroups);
    }

    private List<Queue> uploadQueues(final Connection connection)
        throws SQLException
    {
        int maxId = 0;
        Map<Integer, List<QueueHost>> queuesMap = new HashMap<>();
        try (PreparedStatement statement =
            connection.prepareStatement(QUEUE_QUERY))
        {
            ResultSet resultSet = statement.executeQuery();
            while (resultSet.next()) {
                int id = resultSet.getInt("id");
                String hostname = resultSet.getString("hostname");
                int httpPort = resultSet.getInt("http_port");
                int zkPort = resultSet.getInt("zk_port");
                queuesMap.computeIfAbsent(id, group -> new ArrayList<>())
                    .add(new QueueHost(hostname, zkPort, httpPort));
                if (maxId < id) {
                    maxId = id;
                }
            }
        }
        Queue[] queues = new Queue[maxId + 1];
        for (Map.Entry<Integer, List<QueueHost>> entry: queuesMap.entrySet()) {
            queues[entry.getKey()] = new Queue(entry.getKey(), entry.getValue());
        }
        return Arrays.asList(queues);
    }

    private List<HostGroup> uploadHostGroups(final Connection connection)
        throws SQLException
    {
        int maxId = 0;
        Map<Integer, Set<BackendHost>> hostGroupsMap = new HashMap<>();
        try (PreparedStatement statement =
            connection.prepareStatement(HOST_GROUP_QUERY))
        {
            ResultSet resultSet = statement.executeQuery();
            while (resultSet.next()) {
                int id = resultSet.getInt("id");
                String hostname = resultSet.getString("hostname");
                int searchPort = resultSet.getInt("search_port");
                int searchPortNg = resultSet.getInt("search_port_ng");
                int indexPort = resultSet.getInt("index_port");
                int dumpPort = resultSet.getInt("dump_port");
                int queueIdPort = resultSet.getInt("queue_id_port");
                boolean freshness = resultSet.getBoolean("freshness");
                String dc = resultSet.getString("dc");
                hostGroupsMap.computeIfAbsent(id, group -> new HashSet<>())
                    .add(new BackendHost(
                        hostname,
                        searchPort,
                        searchPortNg,
                        indexPort,
                        dumpPort,
                        queueIdPort,
                        freshness,
                        dc));
                if (maxId < id) {
                    maxId = id;
                }
            }
        }
        HostGroup[] hostGroups = new HostGroup[maxId + 1];
        for (Map.Entry<Integer, Set<BackendHost>> entry:
            hostGroupsMap.entrySet())
        {
            hostGroups[entry.getKey()] =
                new HostGroup(entry.getKey(), entry.getValue());
        }
        return Arrays.asList(hostGroups);
    }

    private SearchMap uploadSearchMap(
        final Connection connection,
        final List<Queue> queues,
        final List<HostGroup> hostGroups)
        throws SQLException
    {
        int maxVersion = 0;
        Map<Metashard, List<Integer>> metashardsMap = new HashMap<>();
        try (PreparedStatement statement =
            connection.prepareStatement(METASHARD_QUERY))
        {
            ResultSet resultSet = statement.executeQuery();
            while (resultSet.next()) {
                String serviceStr = resultSet.getString("service")
                    .toUpperCase(Locale.ROOT);
                Service service;
                try {
                    service = Service.valueOf(serviceStr);
                } catch (IllegalArgumentException e) {
                    throw new SQLException(
                        "Failed to parse service with value " + serviceStr);
                }
                int shard = resultSet.getInt("shard");
                int label = resultSet.getInt("label");
                int queueId = resultSet.getInt("queue_id");
                int hostGroupId = resultSet.getInt("host_group_id");
                int version = resultSet.getInt("version");
                if (version > maxVersion) {
                    maxVersion = version;
                }
                Metashard metashard =
                    new Metashard(label, service, queueId, hostGroupId, null);
                metashardsMap.putIfAbsent(metashard, new ArrayList<>(SHARDS_COUNT));
                metashardsMap.get(metashard).add(shard);
            }
        }
        List<Metashard> metashards = new ArrayList<>();
        for (Map.Entry<Metashard, List<Integer>> entry: metashardsMap.entrySet()) {
            Metashard key = entry.getKey();
            List<ShardsRange> ranges =
                Metashard.generateShardsRanges(entry.getValue());
            for (ShardsRange range: ranges) {
                metashards.add(new Metashard(
                    key.label(),
                    key.service(),
                    key.queueId(),
                    key.hostGroupId(),
                    range));
            }
        }
        return new SearchMap(maxVersion, queues, hostGroups, metashards);
    }
}
