package ru.yandex.search.mop.common.parsers;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.services.Service;

public class SearchMapJsonParser {
    public static final SearchMapJsonParser INSTANCE = new SearchMapJsonParser();

    private final SimpleJsonParser parser = new SimpleJsonParser();

    private SearchMapJsonParser() {
    }

    public SearchMap parse(final String str) throws JsonException {
        JsonObject root = TypesafeValueContentHandler.parse(str);
        return parse(root);
    }

    public SearchMap parse(final Reader reader)
        throws JsonException, IOException
    {
        JsonObject root = parser.parse(reader);
        return parse(root);
    }

    public SearchMap parse(final JsonObject root) throws JsonException {
        JsonMap map = root.asMap();
        if (map.isEmpty()) {
            return null;
        }
        long version = map.getLong("version");
        JsonList queuesJson = map.getList("queues");
        List<Queue> queues = parseQueues(queuesJson);
        if (queues == null) {
            throw new JsonException(
                "Parameter queues with value '" + queuesJson
                    + "' is not valid ");
        }

        JsonList hostGroupsJson = map.getList("host_groups");
        List<HostGroup> hostGroups = parseHostGroups(hostGroupsJson);
        if (hostGroups == null) {
            throw new JsonException(
                "Parameter hostGroups with value '" + hostGroupsJson
                    + "' is not valid ");
        }
        JsonList metashardsJson = map.getListOrNull("metashards");
        List<Metashard> metashards = parseMetashards(metashardsJson);
        if (metashards == null) {
            throw new JsonException(
                "Parameter metashards with value '" + metashardsJson
                    + "' is not valid ");
        }
        return new SearchMap(version, queues, hostGroups, metashards);
    }

    private List<Queue> parseQueues(final JsonList queuesJson)
        throws JsonException
    {
        if (queuesJson == null || queuesJson.isEmpty()) {
            return null;
        }
        Queue[] queues = new Queue[queuesJson.size()];
        for (JsonObject queueObj: queuesJson) {
            JsonMap queueMap = queueObj.asMap();
            int id = (int) queueMap.getLong("id");
            JsonList hosts = queueMap.getList("hosts");
            List<QueueHost> queueHosts = new ArrayList<>();
            for (JsonObject hostObj: hosts) {
                JsonMap hostMap = hostObj.asMap();
                String hostname = hostMap.get("hostname").asString();
                int zkPort = (int) hostMap.getLong("zk_port");
                int httpPort = (int) hostMap.getLong("http_port");
                queueHosts.add(new QueueHost(hostname, zkPort, httpPort));
            }
            queues[id] = new Queue(id, queueHosts);
        }
        return Arrays.asList(queues);
    }

    private List<HostGroup> parseHostGroups(final JsonList hostGroupsJson)
        throws JsonException
    {
        if (hostGroupsJson == null || hostGroupsJson.isEmpty()) {
            return null;
        }
        HostGroup[] hostGroups = new HostGroup[hostGroupsJson.size()];
        for (JsonObject hostGroupObj: hostGroupsJson) {
            JsonMap hostGroupMap = hostGroupObj.asMap();
            int id = (int) hostGroupMap.getLong("id");
            JsonList hosts = hostGroupMap.getList("hosts");
            Set<BackendHost> backendHosts = new HashSet<>();
            for (JsonObject hostObj: hosts) {
                JsonMap hostMap = hostObj.asMap();
                String hostname = hostMap.get("hostname").asString();
                long searchPortNg = hostMap.getLong("search_port_ng");
                long searchPort = hostMap.getLong("search_port");
                long indexPort = hostMap.getLong("index_port");
                long dumpPort = hostMap.getLong("dump_port");

                // TODO remove default value after searchmap updating
                long queueIdPort = hostMap.getLong("queue_id_port", indexPort);

                BackendHost host = new BackendHost(
                    hostname,
                    (int) searchPort,
                    (int) searchPortNg,
                    (int) indexPort,
                    (int) dumpPort,
                    (int) queueIdPort);
                backendHosts.add(host);
            }
            hostGroups[id] = new HostGroup(id, backendHosts);
        }
        return Arrays.asList(hostGroups);
    }

    private List<Metashard> parseMetashards(final JsonList metashardsJson)
        throws JsonException
    {
        if (metashardsJson == null || metashardsJson.isEmpty()) {
            return null;
        }
        List<Metashard> metashards = new ArrayList<>();
        for (JsonObject metashardObj: metashardsJson) {
            JsonMap metashardMap = metashardObj.asMap();
            int label = (int) metashardMap.getLong("label");
            String serviceStr =
                metashardMap.get("service").asString().toUpperCase(Locale.ROOT);
            Service service;
            try {
                service = Service.valueOf(serviceStr);
            } catch (IllegalArgumentException e) {
                throw new JsonException(
                    "Failed to parse service with value " + serviceStr);
            }
            int queueId = (int) metashardMap.getLong("queue_id");
            int hostGroupId = (int) metashardMap.getLong("host_group_id");
            int shardsRangeStart = (int) metashardMap.getLong("shards_range_start");
            int shardsRangeEnd = (int) metashardMap.getLong("shards_range_end");
            Metashard metashard = new Metashard(
                label,
                service,
                queueId,
                hostGroupId,
                shardsRangeStart,
                shardsRangeEnd);
            metashards.add(metashard);
        }
        return metashards;
    }

    private static class SimpleJsonParser {
        private final JsonParser parser;
        private final BasicGenericConsumer<JsonObject, JsonException> consumer;

        public SimpleJsonParser() {
            consumer = new BasicGenericConsumer<>();
            parser =
                new JsonParser(
                    new StackContentHandler(
                        new TypesafeValueContentHandler(consumer)));
        }

        public JsonObject parse(final Reader reader)
            throws JsonException, IOException
        {
            parser.parse(reader);
            return consumer.get();
        }
    }
}
