package ru.yandex.search.mop.common.parsers;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.searchmap.ShardsRange;
import ru.yandex.search.mop.common.services.Service;

public class SearchMapStringParser {
    public static final SearchMapStringParser INSTANCE =
        new SearchMapStringParser();

    private static final String SEARCHMAP_LINE_REGEX =
        "(?<service>[a-z_\\d]+) "
        + "iNum:(?<label>\\d+),"
        + "tag:[a-z-.\\d]+,"
        + "host:(?<host>[a-z-.\\d]+),"
        + "shards:(?<shardsRangeStart>\\d+)-(?<shardsRangeEnd>\\d+),"
        + "zk:(?<zk>[a-z\\-.\\d:/|]+),"
        + "json_indexer_port:(?<jsonIndexerPort>\\d+),"
        + "search_port_ng:(?<searchPortNg>\\d+),"
        + "search_port:(?<searchPort>\\d+),"
        + "dump_port:(?<dumpPort>\\d+)";

    private static final String QUEUE_REGEX =
        "(?<hostname>[a-z-.\\d]+):"
        + "(?<zkPort>\\d+)/"
        + "(?<httpPort>\\d+)";

    private static final String COMMENT = "#";

    private final Pattern searchMapPattern;
    private final Pattern queuePattern;

    private SearchMapStringParser() {
        searchMapPattern = Pattern.compile(SEARCHMAP_LINE_REGEX);
        queuePattern = Pattern.compile(QUEUE_REGEX);
    }

    public SearchMap parse(final Reader reader)
        throws IOException, NumberFormatException
    {
        List<Queue> queues = new ArrayList<>();
        List<HostGroup> hostGroups = new ArrayList<>();
        List<Metashard> metashards = new ArrayList<>();
        BufferedReader bufferedReader = new BufferedReader(reader);
        String line;
        while ((line = bufferedReader.readLine()) != null) {
            line = line.strip();
            if (!line.isEmpty()) {
                try {
                    parseLine(line, queues, metashards, hostGroups);
                } catch (IOException | IllegalArgumentException e) {
                    throw new IOException(
                        "Can't parse searchmap line: " + line, e);
                }
            }
        }
        long version = System.currentTimeMillis();
        return new SearchMap(version, queues, hostGroups, metashards);
    }

    public void parseLine(
        final String line,
        final List<Queue> queues,
        final List<Metashard> metashards,
        final List<HostGroup> hostGroups)
        throws IOException, IllegalArgumentException
    {
        if (line.startsWith(COMMENT)) {
            return;
        }
        Matcher matcher = searchMapPattern.matcher(line);
        if (!matcher.find()) {
            throw new IOException("Searchmap line doesn't match pattern");
        }
        String serviceStr = matcher.group("service").toUpperCase(Locale.ROOT);
        Service service = Service.valueOf(serviceStr);
        int label = Integer.parseInt(matcher.group("label"));
        String hostname = matcher.group("host");
        int shardsRangeStart = Integer.parseInt(matcher.group("shardsRangeStart"));
        int shardsRangeEnd = Integer.parseInt(matcher.group("shardsRangeEnd"));
        String zk = matcher.group("zk");
        int indexerPort = Integer.parseInt(matcher.group("jsonIndexerPort"));
        int searchPortNg = Integer.parseInt(matcher.group("searchPortNg"));
        int searchPort = Integer.parseInt(matcher.group("searchPort"));
        int dumpPort = Integer.parseInt(matcher.group("dumpPort"));

        int queueId = parseQueue(zk, queues);
        ShardsRange shardsRange =
            new ShardsRange(shardsRangeStart, shardsRangeEnd);
        BackendHost host = new BackendHost(
            hostname,
            searchPort,
            searchPortNg,
            indexerPort,
            dumpPort,
            indexerPort);
        for (Metashard metashard: metashards) {
            if (metashard.service().equals(service)
                && metashard.label() == label
                && metashard.queueId() == queueId
                && metashard.shardsRange().equals(shardsRange))
            {
                HostGroup hostGroup = hostGroups.get(metashard.hostGroupId());
                hostGroup.hosts().add(host);
                return;
            }
        }
        int hostGroupId = -1;
        for (HostGroup hostGroup: hostGroups) {
            if (hostGroup.hosts().contains(host)) {
                hostGroupId = hostGroup.id();
                break;
            }
        }
        if (hostGroupId == -1) {
            hostGroupId = hostGroups.size();
            Set<BackendHost> hosts = new HashSet<>();
            hosts.add(host);
            hostGroups.add(new HostGroup(hostGroupId, hosts));
        }
        Metashard metashard =
            new Metashard(label, service, queueId, hostGroupId, shardsRange);
        metashards.add(metashard);
    }

    public int parseQueue(final String line, final List<Queue> queues)
        throws IOException
    {
        List<QueueHost> hosts = new ArrayList<>();
        Matcher matcher = queuePattern.matcher(line);
        while (matcher.find()) {
            String hostname = matcher.group("hostname");
            int zkPort = Integer.parseInt(matcher.group("zkPort"));
            int httpPort = Integer.parseInt(matcher.group("httpPort"));
            QueueHost host = new QueueHost(hostname, zkPort, httpPort);
            hosts.add(host);
        }
        if (hosts.isEmpty()) {
            throw new IOException("Can't parse queue: " + line);
        }
        int index = queues.size();
        Queue queue = new Queue(index, hosts);
        for (Queue existingQueue: queues) {
            if (queue.equals(existingQueue)) {
                return existingQueue.id();
            }
        }
        queues.add(queue);
        return index;
    }
}
