package ru.yandex.search.mop.common.writers;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.searchmap.ShardsRange;

public class SearchMapDBWriter {
    public static final SearchMapDBWriter INSTANCE =
        new SearchMapDBWriter();

    public static final int BATCH_SIZE = 1;

    public static final String QUEUE_INSERT_QUERY =
        "INSERT INTO queue (id, hostname, http_port, zk_port)"
            + "VALUES (?, ?, ?, ?)";
    public static final String HOST_GROUP_INSERT_QUERY =
        "INSERT INTO host_group (id, hostname, search_port, search_port_ng,"
            + "index_port, dump_port, queue_id_port, freshness, dc)"
            + "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    public static final String METASHARD_INSERT_QUERY =
        "INSERT INTO metashard (service, shard, label, queue_id, host_group_id,"
            + "version)"
            + "VALUES (?, ?, ?, ?, ?, ?)";

    public static final String VLA = "vla";
    public static final String MAN = "man";
    public static final String SAS = "sas";
    public static final String MYT = "myt";
    public static final String IVA = "iva";

    private SearchMapDBWriter() {
    }

    public void write(final Connection connection, final SearchMap searchMap)
        throws ClassNotFoundException, SQLException
    {
        writeQueues(connection, searchMap.queues());
        writeHostGroups(connection, searchMap.hostGroups());
        writeMetashards(connection, searchMap.metashards());
    }

    private void writeQueues(
        final Connection connection,
        final List<Queue> queues)
        throws SQLException
    {
        try (PreparedStatement statement =
            connection.prepareStatement(QUEUE_INSERT_QUERY))
        {
            int count = 0;
            for (Queue queue: queues) {
                for (QueueHost host: queue.hosts()) {
                    statement.setInt(1, queue.id());
                    statement.setString(2, host.hostname());
                    statement.setInt(3, host.httpPort());
                    statement.setInt(4, host.zkPort());
                    statement.addBatch();
                    if (++count % BATCH_SIZE == 0) {
                        statement.executeBatch();
                        statement.clearBatch();
                    }
                }
            }
            if (count % BATCH_SIZE != 0) {
                statement.executeBatch();
                statement.clearBatch();
            }
        }
    }

    private void writeHostGroups(
        final Connection connection,
        final List<HostGroup> hostGroups)
        throws SQLException
    {
        try (PreparedStatement statement =
            connection.prepareStatement(HOST_GROUP_INSERT_QUERY))
        {
            int count = 0;
            for (HostGroup hostGroup: hostGroups) {
                for (BackendHost host: hostGroup.hosts()) {

                    // TODO delete after moving to the DB
                    String dc = host.dc();
                    if (dc == null) {
                        dc = extractDc(host.hostname());
                    }

                    statement.setInt(1, hostGroup.id());
                    statement.setString(2, host.hostname());
                    statement.setInt(3, host.searchPort());
                    statement.setInt(4, host.searchPortNg());
                    statement.setInt(5, host.indexPort());
                    statement.setInt(6, host.dumpPort());
                    statement.setInt(7, host.queueIdPort());
                    statement.setBoolean(8, host.freshness());
                    statement.setString(9, dc);
                    statement.addBatch();
                    if (++count % BATCH_SIZE == 0) {
                        statement.executeBatch();
                    }
                }
            }
            statement.executeBatch();
        }
    }

    private void writeMetashards(
        final Connection connection,
        final List<Metashard> metashards)
        throws SQLException
    {
        try (PreparedStatement statement =
            connection.prepareStatement(METASHARD_INSERT_QUERY))
        {
            int count = 0;
            for (Metashard metashard: metashards) {
                ShardsRange range = metashard.shardsRange();
                for (int shard = range.start(); shard <= range.end(); shard++) {
                    statement.setString(1, metashard.service().lowName());
                    statement.setInt(2, shard);
                    statement.setInt(3, metashard.label());
                    statement.setInt(4, metashard.queueId());
                    statement.setInt(5, metashard.hostGroupId());
                    statement.setInt(6, 0);
                    statement.addBatch();
                    if (++count % BATCH_SIZE == 0) {
                        statement.executeBatch();
                    }
                }
            }
            statement.executeBatch();
        }
    }

    private String extractDc(final String hostname) {
        String dc = null;
        if (hostname.contains(VLA)) {
            dc = "vla";
        } else if (hostname.contains(SAS)) {
            dc = "sas";
        } else if (hostname.contains(MAN)) {
            dc = "man";
        } else if (hostname.contains(MYT)) {
            dc = "myt";
        } else if (hostname.contains(IVA)) {
            dc = "iva";
        }
        return dc;
    }
}
