package ru.yandex.search.mop.common.writers;

import java.io.IOException;

import ru.yandex.http.util.NotFoundException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.QueueHost;
import ru.yandex.search.mop.common.searchmap.SearchMap;

public class SearchMapJsonWriter implements SearchMapWriter {
    public static final SearchMapJsonWriter INSTANCE =
        new SearchMapJsonWriter();

    private SearchMapJsonWriter() {
    }

    public StringBuilderWriter write(final SearchMap searchMap)
        throws IOException, NotFoundException
    {
        return write(searchMap, JsonType.NORMAL);
    }

    public StringBuilderWriter write(
        final SearchMap searchMap,
        final JsonType jsonType)
        throws IOException, NotFoundException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = jsonType.create(sbw)) {
            writer.startObject();
            writer.key("version");
            writer.value(searchMap.version());
            writeQueues(writer, searchMap);
            writeHostGroups(writer, searchMap);
            writeMetashards(writer, searchMap);
            writer.endObject();
        }
        return sbw;
    }

    private void writeQueues(final JsonWriter writer, final SearchMap searchMap)
        throws IOException
    {
        writer.key("queues");
        writer.startArray();
        for (Queue queue: searchMap.queues()) {
            writer.startObject();
            writer.key("id");
            writer.value(queue.id());
            writer.key("hosts");
            writer.startArray();
            for (QueueHost host: queue.hosts()) {
                writer.startObject();
                writer.key("hostname");
                writer.value(host.hostname());
                writer.key("zk_port");
                writer.value(host.zkPort());
                writer.key("http_port");
                writer.value(host.httpPort());
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        }
        writer.endArray();
    }

    private void writeHostGroups(
        final JsonWriter writer,
        final SearchMap searchMap)
        throws IOException, NotFoundException
    {
        writer.key("host_groups");
        writer.startArray();
        for (HostGroup hostGroup: searchMap.hostGroups()) {
            writer.startObject();
            writer.key("id");
            writer.value(hostGroup.id());
            writer.key("hosts");
            writer.startArray();
            for (BackendHost host: searchMap.hosts(hostGroup.id())) {
                writer.startObject();
                writer.key("hostname");
                writer.value(host.hostname());
                writer.key("search_port_ng");
                writer.value(host.searchPortNg());
                writer.key("search_port");
                writer.value(host.searchPort());
                writer.key("index_port");
                writer.value(host.indexPort());
                writer.key("dump_port");
                writer.value(host.dumpPort());
                writer.key("queue_id_port");
                writer.value(host.queueIdPort());
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        }
        writer.endArray();
    }

    private void writeMetashards(
        final JsonWriter writer,
        final SearchMap searchMap)
        throws IOException
    {
        writer.key("metashards");
        writer.startArray();
        for (Metashard metashard: searchMap.metashards()) {
            writer.startObject();
            writer.key("label");
            writer.value(metashard.label());
            writer.key("service");
            writer.value(metashard.service().lowName());
            writer.key("queue_id");
            writer.value(metashard.queueId());
            writer.key("host_group_id");
            writer.value(metashard.hostGroupId());
            writer.key("shards_range_start");
            writer.value(metashard.shardsRange().start());
            writer.key("shards_range_end");
            writer.value(metashard.shardsRange().end());
            writer.endObject();
        }
        writer.endArray();
    }
}
