package ru.yandex.search.mop.common.writers;

import ru.yandex.http.util.NotFoundException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.SearchMap;

public class SearchMapStringWriter implements SearchMapWriter {
    public static final SearchMapStringWriter INSTANCE =
        new SearchMapStringWriter();

    private static final String GENCFG_SUFFIX = ".gencfg-c.yandex.net";
    private static final String YP_SUFFIX = ".yp-c.yandex.net";

    private SearchMapStringWriter() {
    }

    public StringBuilderWriter write(final SearchMap searchMap)
        throws NotFoundException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        for (Metashard metashard: searchMap.metashards()) {
            sbw.append(writeMetashard(searchMap, metashard).sb());
        }
        return sbw;
    }

    public StringBuilderWriter writeMetashard(
        final SearchMap searchMap,
        final Metashard metashard)
        throws NotFoundException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        for (BackendHost host: searchMap.hosts(metashard.hostGroupId())) {
            sbw.append(writeHost(searchMap, metashard, host).sb());
        }
        return sbw;
    }

    public StringBuilderWriter writeHost(
        final SearchMap searchMap,
        final Metashard metashard,
        final BackendHost host)
        throws NotFoundException
    {
        Queue queue = searchMap.queues().get(metashard.queueId());
        if (queue == null) {
            throw new NotFoundException(
                "Queue not found: " + metashard.queueId());
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        sbw.append(metashard.service().lowName()).append(' ');
        sbw.append("iNum:").print(metashard.label());
        sbw.append(",tag:").append(tagFromHost(host.hostname()));
        sbw.append(",host:").append(host.hostname());
        sbw.append(",shards:").append(metashard.shardsRange().toString());
        sbw.append(",zk:").append(queue.queueStr());
        sbw.append(",json_indexer_port:").print(host.indexPort());
        sbw.append(",search_port_ng:").print(host.searchPortNg());
        sbw.append(",search_port:").print(host.searchPort());
        sbw.append(",dump_port:").print(host.dumpPort());
        sbw.append(",queue_id_port:").print(host.queueIdPort());
        sbw.append('\n');
        return sbw;
    }

    private String tagFromHost(final String host) {
        return host.replace(GENCFG_SUFFIX, "").replace(YP_SUFFIX, "");
    }
}
