package ru.yandex.search.mop.server;

import java.sql.SQLException;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.services.Service;
import ru.yandex.search.mop.common.services.ServiceScope;
import ru.yandex.search.mop.common.writers.SearchMapStringWriter;
import ru.yandex.search.mop.server.searchmap.ServerSearchMap;

public class GetShardsForConsumeHandler implements ProxyRequestHandler {
    private final MopServer mop;

    public GetShardsForConsumeHandler(final MopServer mop) {
        this.mop = mop;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException
    {
        GetShardsForConsumeContext context =
            new GetShardsForConsumeContext(session.params(), session.logger());
        if (mop.config().pumpkin()) {
            context.logger().log(Level.SEVERE, "Pumpkin is activated");
            session.response(
                HttpStatus.SC_NOT_IMPLEMENTED,
                new NStringEntity(
                    "Pumpkin is activated",
                    ContentType.TEXT_PLAIN
                        .withCharset(session.acceptedCharset())));
            return;
        }
        ServiceScope scope = context.scope();
        int label = context.label();
        ServerSearchMap searchMap = mop.searchMapContainer().searchMap();

        mop.searchMapContainer().writeLock().lock();
        try {
            Integer prevLabel = searchMap.labelByHost(scope, context.host());
            if (prevLabel != null && prevLabel != label) {
                String msg = "Host already exists in label " + prevLabel;
                context.logger().log(Level.SEVERE, msg);
                session.response(
                    HttpStatus.SC_BAD_REQUEST,
                    new NStringEntity(
                        msg,
                        ContentType.TEXT_PLAIN
                            .withCharset(session.acceptedCharset())));
                return;
            }
            try {
                for (Service service: scope.services()) {
                    Integer hostGroupId =
                        mop.metashardDAO().getHostGroupId(service, label);
                    BackendHost newHost = new BackendHost(
                        context.host(),
                        context.dc(),
                        context.port(),
                        service.config());
                    if (searchMap.containsHost(hostGroupId, newHost)) {
                        continue;
                    }
                    int version =
                        mop.hostGroupDAO().insertHostGroup(hostGroupId, newHost);
                    if (version >= 0) {
                        searchMap.addHost(scope, version, hostGroupId, label, newHost);
                        context.logger().info(
                            "Host " + newHost + " added for label " + label
                                + ", for service " + service
                                + ", for host group " + hostGroupId
                                + "; new searchmap version: "
                                + searchMap.version());
                    }
                }
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        SearchMapStringWriter.INSTANCE.write(
                            searchMap.getPartByScope(scope)).toString(),
                        ContentType.TEXT_PLAIN
                            .withCharset(session.acceptedCharset())));
            } catch (SQLException e) {
                context.logger().log(Level.SEVERE, "Database error", e);
                session.response(
                    HttpStatus.SC_INTERNAL_SERVER_ERROR,
                    new NStringEntity(
                        e.getMessage(),
                        ContentType.TEXT_PLAIN
                            .withCharset(session.acceptedCharset())));
            } catch (NotFoundException e) {
                context.logger().log(Level.SEVERE, "Not found exception", e);
                session.handleException(e);
            }
        } finally {
            mop.searchMapContainer().writeLock().unlock();
        }
    }
}
