package ru.yandex.search.mop.server;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Timer;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.parser.JsonException;
import ru.yandex.search.mop.server.config.ImmutableMopServerConfig;
import ru.yandex.search.mop.server.dao.HostGroupDAO;
import ru.yandex.search.mop.server.dao.MetashardDAO;
import ru.yandex.search.mop.server.dao.QueueDAO;
import ru.yandex.search.mop.server.dao.SearchMapDAO;
import ru.yandex.search.mop.server.pool.ConnectionPool;
import ru.yandex.search.mop.server.searchmap.ServerSearchMapContainer;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class MopServer
    extends HttpProxy<ImmutableMopServerConfig>
    implements ProxyRequestHandler
{
    private final static int FLUSH_INTERVAL = 172800000; // every 2 days

    private final TimeFrameQueue<Long> getSearchMapActualVersion;

    private ServerSearchMapContainer searchMapContainer;

    private UpdateSearchMapTask updateSearchMapTask;
    private Timer updateSearchMapTimer;
    private FlushSearchMapTask flushSearchMapTask;
    private Timer flushSearchMapTimer;

    private ConnectionPool connectionPool;
    private HostGroupDAO hostGroupDAO;
    private MetashardDAO metashardDAO;
    private QueueDAO queueDAO;
    private SearchMapDAO searchMapDAO;

    public MopServer(ImmutableMopServerConfig config) throws IOException {
        super(config);
        GetSearchMapHandler getSearchMapHandler =
            new GetSearchMapHandler(this);
        GetSearchHostsHandler getSearchHostsHandler =
            new GetSearchHostsHandler(this);
        GetShardsForConsumeHandler getShardsForConsumeHandler =
            new GetShardsForConsumeHandler(this);
        MoveShardHandler moveShardHandler =
            new MoveShardHandler(this);
        DeleteSearchHostsHandler deleteSearchHostsHandler =
            new DeleteSearchHostsHandler(this);
        register(new Pattern<>("", true), this, "POST");
        register(
            new Pattern<>("/get-searchmap", true),
            getSearchMapHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-search-hosts", true),
            getSearchHostsHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-shards-for-consume", true),
            getShardsForConsumeHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/move-shard", true),
            moveShardHandler,
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/delete-search-hosts", true),
            deleteSearchHostsHandler,
            RequestHandlerMapper.GET);

        getSearchMapActualVersion =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                getSearchMapActualVersion,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "get-searchmap-actual-version_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "get-searchmap-total-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));
        connectionPool(new ConnectionPool(config.databaseConfig()));
        updateSearchMapTask = new UpdateSearchMapTask(this);
        updateSearchMapTimer = new Timer("Update-searchmap-timer");
        flushSearchMapTask = new FlushSearchMapTask(this);
        flushSearchMapTimer = new Timer("Flush-searchmap-timer");
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        session.response(HttpStatus.SC_OK);
    }

    public ServerSearchMapContainer searchMapContainer() {
        return searchMapContainer;
    }

    public ConnectionPool connectionPool() {
        return connectionPool;
    }

    public void connectionPool(final ConnectionPool connectionPool) {
        this.connectionPool = connectionPool;
        this.hostGroupDAO = new HostGroupDAO(connectionPool);
        this.metashardDAO = new MetashardDAO(connectionPool);
        this.queueDAO = new QueueDAO(connectionPool);
        this.searchMapDAO = new SearchMapDAO(connectionPool);
    }

    public HostGroupDAO hostGroupDAO() {
        return hostGroupDAO;
    }

    public MetashardDAO metashardDAO() {
        return metashardDAO;
    }

    public QueueDAO queueDAO() {
        return queueDAO;
    }

    public SearchMapDAO searchMapDAO() {
        return searchMapDAO;
    }

    public void getSearchMapResponse(final boolean isVersionActual) {
        getSearchMapActualVersion.accept(isVersionActual? 1L : 0L);
    }

    @Override
    public void start() throws IOException {
        if (config.pumpkin()) {
            try {
                this.searchMapContainer =
                    new ServerSearchMapContainer(config.file());
            } catch (JsonException e) {
                throw new IOException(e);
            }
            super.start();
            return;
        }

        try {
            this.searchMapContainer = new ServerSearchMapContainer(this);
        } catch (SQLException e) {
            throw new IOException(e);
        }
        updateSearchMapTimer.schedule(
            updateSearchMapTask,
            config.interval(),
            config.interval());
        flushSearchMapTimer.schedule(
            flushSearchMapTask,
            FLUSH_INTERVAL,
            FLUSH_INTERVAL);
        super.start();
    }
}
