package ru.yandex.search.mop.server.dao;

import java.sql.Array;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.server.pool.ConnectionPool;

public class HostGroupDAO {
    public static final String SELECT_ALL =
        "SELECT * FROM host_group";

    public static final String INSERT_HOST_GROUP =
        "INSERT INTO host_group(id, hostname, search_port, search_port_ng, "
            + "index_port, dump_port, queue_id_port, freshness, dc) "
            + "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?) "
            + "ON CONFLICT ON CONSTRAINT uq_host_group_id_hostname do nothing";

    public static final String DELETE_HOSTNAME =
        "DELETE FROM host_group WHERE hostname = ? RETURNING id";

    private final ConnectionPool connectionPool;

    public HostGroupDAO(final ConnectionPool connectionPool) {
        this.connectionPool = connectionPool;
    }

    public static List<HostGroup> parse(final ResultSet resultSet)
        throws SQLException
    {
        int maxId = -1;
        Map<Integer, Set<BackendHost>> hostGroupsMap = new HashMap<>();
        while (resultSet.next()) {
            int id = resultSet.getInt("id");
            String hostname = resultSet.getString("hostname");
            int searchPort = resultSet.getInt("search_port");
            int searchPortNg = resultSet.getInt("search_port_ng");
            int indexPort = resultSet.getInt("index_port");
            int dumpPort = resultSet.getInt("dump_port");
            int queueIdPort = resultSet.getInt("queue_id_port");
            boolean freshness = resultSet.getBoolean("freshness");
            String dc = resultSet.getString("dc");

            BackendHost host = new BackendHost(
                hostname,
                searchPort,
                searchPortNg,
                indexPort,
                dumpPort,
                queueIdPort,
                freshness,
                dc);

            hostGroupsMap.computeIfAbsent(id, group -> new HashSet<>()).add(host);
            if (maxId < id) {
                maxId = id;
            }
        }
        HostGroup[] hostGroups = new HostGroup[maxId + 1];
        for (Map.Entry<Integer, Set<BackendHost>> entry: hostGroupsMap.entrySet()) {
            hostGroups[entry.getKey()] =
                new HostGroup(entry.getKey(), entry.getValue());
        }
        return Arrays.asList(hostGroups);
    }

    public int insertHostGroup(final Integer id, final BackendHost host)
        throws SQLException
    {
        try (Connection connection = connectionPool.getConnection()) {
            try (PreparedStatement insert = connection.prepareStatement(
                    INSERT_HOST_GROUP);
                PreparedStatement versionInc = connection.prepareStatement(
                    MetashardDAO.INCREMENT_VERSION_BY_HOST_GROUP))
            {
                connection.setAutoCommit(false);

                insert.setInt(1, id);
                insert.setString(2, host.hostname());
                insert.setInt(3, host.searchPort());
                insert.setInt(4, host.searchPortNg());
                insert.setInt(5, host.indexPort());
                insert.setInt(6, host.dumpPort());
                insert.setInt(7, host.queueIdPort());
                insert.setBoolean(8, host.freshness());
                insert.setString(9, host.dc());
                int inserted = insert.executeUpdate();
                if (inserted > 0) {
                    versionInc.setInt(1, id);
                    return executeIncVersionStatement(connection, versionInc);
                } else {
                    connection.rollback();
                    return -1;
                }
            } catch (SQLException e) {
                connection.rollback();
                throw e;
            } finally {
                if (connection != null) {
                    connection.setAutoCommit(true);
                }
            }
        }
    }

    public int deleteHostname(final String hostname) throws SQLException {
        try (Connection connection = connectionPool.getConnection()) {
            try (PreparedStatement delete =
                     connection.prepareStatement(DELETE_HOSTNAME);
                 PreparedStatement versionInc = connection.prepareStatement(
                     MetashardDAO.INCREMENT_VERSION_BY_FEW_HOST_GROUPS))
            {
                connection.setAutoCommit(false);

                delete.setString(1, hostname);
                ResultSet resultSet = delete.executeQuery();
                Set<Integer> groupIds = new HashSet<>();
                while(resultSet.next()) {
                    groupIds.add(resultSet.getInt("id"));
                }
                if (groupIds.size() > 0) {
                    Array array = connection.createArrayOf("int", groupIds.toArray());
                    versionInc.setArray(1, array);
                    return executeIncVersionStatement(connection, versionInc);
                } else {
                    connection.commit();
                    return 0;
                }
            } catch (SQLException e) {
                connection.rollback();
                throw e;
            } finally {
                if (connection != null) {
                    connection.setAutoCommit(true);
                }
            }
        }
    }

    private int executeIncVersionStatement(
        final Connection connection,
        final PreparedStatement statement)
        throws SQLException
    {
        statement.execute();
        ResultSet resultSet = statement.getResultSet();
        if (resultSet != null && resultSet.next()) {
            int version = resultSet.getInt("nextval");
            connection.commit();
            return version;
        } else {
            connection.rollback();
            return -1;
        }
    }
}
