package ru.yandex.search.mop.server.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.ShardsRange;
import ru.yandex.search.mop.common.services.Service;
import ru.yandex.search.mop.server.pool.ConnectionPool;

public class MetashardDAO {
    public static final String SELECT_ALL =
        "SELECT * FROM metashard;";

    public static final String SELECT_HOST_GROUP_ID =
        "SELECT DISTINCT host_group_id "
            + "FROM metashard WHERE service = ? AND label = ?;";

    public static final String SELECT_MAX_VERSION =
        "SELECT MAX(version) FROM metashard;";

    public static final String INCREMENT_VERSION_BY_HOST_GROUP =
        "SELECT NEXTVAL('metashard_versions');"
            + "UPDATE metashard SET version = CURRVAL('metashard_versions') "
            + "WHERE host_group_id = ?";

    public static final String INCREMENT_VERSION_BY_FEW_HOST_GROUPS =
        "SELECT NEXTVAL('metashard_versions');"
            + "UPDATE metashard SET version = CURRVAL('metashard_versions') "
            + "WHERE host_group_id = ANY (?)";

    private final ConnectionPool connectionPool;

    public MetashardDAO(final ConnectionPool connectionPool) {
        this.connectionPool = connectionPool;
    }

    public static List<Metashard> parse(final ResultSet resultSet)
        throws SQLException
    {
        Map<Metashard, List<Integer>> metashardsMap = new HashMap<>();
        while (resultSet.next()) {
            String serviceStr = resultSet.getString("service")
                .toUpperCase(Locale.ROOT);
            Service service;
            try {
                service = Service.valueOf(serviceStr);
            } catch (IllegalArgumentException e) {
                throw new SQLException(
                    "Failed to parse service with value " + serviceStr);
            }
            int shard = resultSet.getInt("shard");
            int label = resultSet.getInt("label");
            int queueId = resultSet.getInt("queue_id");
            int hostGroupId = resultSet.getInt("host_group_id");
            Metashard metashard =
                new Metashard(label, service, queueId, hostGroupId, null);
            metashardsMap.putIfAbsent(metashard, new ArrayList<>());
            metashardsMap.get(metashard).add(shard);
        }
        List<Metashard> metashards = new ArrayList<>();
        for (Map.Entry<Metashard, List<Integer>> entry: metashardsMap.entrySet()) {
            Metashard key = entry.getKey();
            List<ShardsRange> ranges =
                Metashard.generateShardsRanges(entry.getValue());
            for (ShardsRange range: ranges) {
                metashards.add(new Metashard(
                    key.label(),
                    key.service(),
                    key.queueId(),
                    key.hostGroupId(),
                    range));
            }
        }
        return metashards;
    }

    public static int parseVersion(final ResultSet resultSet)
        throws SQLException
    {
        if (resultSet.next()) {
            return resultSet.getInt("max");
        } else {
            throw new SQLException("Metashard version max is absent");
        }
    }

    public Integer getHostGroupId(final Service service, final int label)
        throws SQLException
    {
        try (Connection connection = connectionPool.getConnection();
            PreparedStatement statement =
                connection.prepareStatement(SELECT_HOST_GROUP_ID))
        {
            statement.setString(1, service.lowName());
            statement.setInt(2, label);
            ResultSet resultSet = statement.executeQuery();
            int id = -1;
            if (resultSet.next()) {
                id = resultSet.getInt("host_group_id");
            }
            if (resultSet.next()) {
                throw new SQLException(
                    "More than one host_group_id for label: " + label
                        + ", and service: " + service.lowName());
            }
            if (id == -1) {
                throw new SQLException(
                    "Not found host_group_id for label: " + label
                        + ", and service: " + service.lowName());
            }
            return id;
        }
    }

    public Integer getMaxVersion() throws SQLException {
        try (Connection connection = connectionPool.getConnection();
            PreparedStatement statement =
                connection.prepareStatement(SELECT_MAX_VERSION))
        {
            return parseVersion(statement.executeQuery());
        }
    }
}
