package ru.yandex.search.mop.server.searchmap;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;

import ru.yandex.http.util.NotFoundException;
import ru.yandex.search.mop.common.searchmap.BackendHost;
import ru.yandex.search.mop.common.searchmap.HostGroup;
import ru.yandex.search.mop.common.searchmap.Metashard;
import ru.yandex.search.mop.common.searchmap.Queue;
import ru.yandex.search.mop.common.searchmap.SearchMap;
import ru.yandex.search.mop.common.services.Service;
import ru.yandex.search.mop.common.services.ServiceScope;

public class ServerSearchMap extends SearchMap {
    private final Map<ServiceScope, Map<String, Integer>> labelByHost;

    private final Lock readLock;

    public ServerSearchMap(final SearchMap searchMap, final Lock readLock) {
        super(
            searchMap.version(),
            searchMap.queues(),
            searchMap.hostGroups(),
            searchMap.metashards());
        labelByHost = generateLabelByHost();
        this.readLock = readLock;
    }

    @Override
    public long version() {
        readLock.lock();
        try {
            return version;
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public List<Queue> queues() {
        readLock.lock();
        try {
            return queues;
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public List<HostGroup> hostGroups() {
        readLock.lock();
        try {
            return hostGroups;
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public Set<BackendHost> hosts(final int hostGroupId)
        throws NotFoundException
    {
        readLock.lock();
        try {
            HostGroup hostGroup = hostGroups.get(hostGroupId);
            if (hostGroup == null) {
                throw new NotFoundException(
                    "Host group not found: " + hostGroupId);
            }
            return hostGroup.hosts();
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public List<Metashard> metashards() {
        readLock.lock();
        try {
            return metashards;
        } finally {
            readLock.unlock();
        }
    }

    public Integer labelByHost(final ServiceScope scope, final String hostname) {
        return labelByHost.get(scope).get(hostname);
    }

    public void addHost(
        final ServiceScope scope,
        final int version,
        final int hostGroupId,
        final int label,
        final BackendHost host)
        throws NotFoundException
    {
        HostGroup hostGroup = hostGroups.get(hostGroupId);
        if (hostGroup == null) {
            throw new NotFoundException(
                "Host group not found: " + hostGroupId);
        }
        hostGroup.hosts().add(host);
        labelByHost.get(scope).put(host.hostname(), label);
        if (version > this.version) {
            this.version = version;
        }
    }

    public boolean containsHost(final Integer hostGroupId, final BackendHost host) {
        return hostGroups.get(hostGroupId).hosts().contains(host);
    }

    public int deleteHost(final String host, final int version) {
        int count = 0;
        for (HostGroup hostGroup: hostGroups) {
            for (Iterator<BackendHost> iter = hostGroup.hosts().iterator(); iter.hasNext(); ) {
                BackendHost backendHost = iter.next();
                if (backendHost.hostname().contains(host)) {
                    for (ServiceScope scope: ServiceScope.values()) {
                        labelByHost.get(scope).remove(backendHost.hostname());
                    }
                    iter.remove();
                    ++count;
                }
            }
        }
        if (version > this.version) {
            this.version = version;
        }
        return count;
    }

    private Map<ServiceScope, Map<String, Integer>> generateLabelByHost() {
        Map<ServiceScope, Map<String, Integer>> map = new HashMap<>();
        Map<Service, ServiceScope> services = new HashMap<>();
        for (ServiceScope scope: ServiceScope.values()) {
            map.put(scope, new HashMap<>());
            for (Service service: scope.services()) {
                services.put(service, scope);
            }
        }
        for (Metashard metashard: metashards) {
            ServiceScope scope = services.get(metashard.service());
            if (scope == null) {
                continue;
            }
            Map<String, Integer> labelsMap = map.get(scope);
            HostGroup hostGroup = hostGroups.get(metashard.hostGroupId());
            for (BackendHost host: hostGroup.hosts()) {
                labelsMap.putIfAbsent(host.hostname(), metashard.label());
            }
        }
        return map;
    }
}
