CREATE TABLE queue
(
    id        int  NOT NULL,
    hostname  text NOT NULL,
    http_port int  NOT NULL,
    zk_port   int  NOT NULL
);


CREATE TABLE host_group
(
    id             int     NOT NULL,
    hostname       text    NOT NULL,
    search_port    int     NOT NULL,
    search_port_ng int     NOT NULL,
    index_port     int     NOT NULL,
    dump_port      int     NOT NULL,
    queue_id_port  int     NOT NULL,
    freshness      boolean NOT NULL,
    dc             text    NOT NULL,
    CONSTRAINT uq_host_group_id_hostname UNIQUE (id, hostname)
);


CREATE TABLE metashard
(
    service       text NOT NULL,
    shard         int  NOT NULL,
    label         int  NOT NULL,
    queue_id      int  NOT NULL,
    host_group_id int  NOT NULL,
    version       int  NOT NULL DEFAULT 0
);


CREATE TABLE action_log
(
    id         serial,
    time       timestamp DEFAULT now(),
    table_name text,
    operation  text,
    new_val    json,
    old_val    json,
    user_name  text
);


CREATE SEQUENCE metashard_versions;


CREATE OR REPLACE FUNCTION process_action_log()
RETURNS TRIGGER AS $action_log$
BEGIN
    IF (TG_OP = 'INSERT') THEN
        INSERT INTO action_log(TABLE_NAME, operation, new_val, old_val, user_name)
        VALUES(TG_TABLE_NAME, 'insert', to_jsonb(NEW), NULL, CURRENT_USER);
        RETURN new;
    ELSIF (TG_OP = 'UPDATE') THEN
        INSERT INTO action_log(TABLE_NAME, operation, new_val, old_val, user_name)
        VALUES(TG_TABLE_NAME, 'update', to_jsonb(NEW), to_jsonb(OLD), CURRENT_USER);
        RETURN new;
    ELSIF (TG_OP = 'DELETE') THEN
        INSERT INTO action_log(TABLE_NAME, operation, new_val, old_val, user_name)
        VALUES(TG_TABLE_NAME, 'delete', NULL, to_jsonb(OLD), CURRENT_USER);
        RETURN old;
    END IF;
END;
$action_log$
LANGUAGE plpgsql;


CREATE TRIGGER action_log
    AFTER INSERT OR DELETE ON metashard
        FOR EACH ROW EXECUTE PROCEDURE process_action_log();


CREATE TRIGGER action_log
    AFTER INSERT OR UPDATE OR DELETE ON host_group
        FOR EACH ROW EXECUTE PROCEDURE process_action_log();


CREATE TRIGGER action_log
    AFTER INSERT OR UPDATE OR DELETE ON queue
        FOR EACH ROW EXECUTE PROCEDURE process_action_log();