package ru.yandex.personal.mail.search.metrics.scraper.controllers;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.controllers.mapping.ArchiveMapping;
import ru.yandex.personal.mail.search.metrics.scraper.services.archive.GenericArchive;

@Service("archiveController")
public class ArchiveController implements ArchiveMapping {
    private final Map<String, GenericArchive> archiveMap = new HashMap<>();

    @Autowired
    public ArchiveController(List<GenericArchive> archives) {
        for (GenericArchive archive : archives) {
            archiveMap.put(archive.getName(), archive);
        }
    }

    @Override
    public List<String> getArchives(HttpServletRequest request) {
        String path = request.getScheme() + "://" + request.getServerName();
        return archiveMap.keySet().stream()
                .map(archiveName -> path + "/api/archive/" + archiveName)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getArchiveDates(HttpServletRequest request, String archiveName) {
        String path = request.getScheme() + "://" + request.getServerName();
        GenericArchive<?> archive = archiveMap.get(archiveName);

        return archive.entryKeys().stream()
                .map(key -> path + "/api/archive/" + archiveName + "/" + key)
                .collect(Collectors.toList());
    }

    @Override
    public Response getArchive(
            String archiveName,
            String resourceId,
            String resourceDate)
    {
        String decoded = URLDecoder.decode(resourceId, StandardCharsets.UTF_8);
        GenericArchive<?> archive = archiveMap.get(archiveName);
        Object responseBody = archive.getContent(resourceDate + "/" + decoded);
        return Response
                .status(Response.Status.OK)
                .entity(responseBody)
                .type(archive.contentType())
                .build();
    }

    @Override
    public Response getScriptlessResponse(
            String archiveName,
            String resourceId,
            String resourceDate)
    {
        String decoded = URLDecoder.decode(resourceId, StandardCharsets.UTF_8);
        GenericArchive<?> archive = archiveMap.get(archiveName);
        Object responseBody = archive.getContent(resourceDate + "/" + decoded);

        if (archive.contentType() == MediaType.TEXT_HTML_TYPE) {
            Document doc = Jsoup.parse((String) responseBody);
            doc.select("script").remove();
            responseBody = doc.toString();
        }

        return Response
                .status(Response.Status.OK)
                .entity(responseBody)
                .type(archive.contentType())
                .build();
    }
}
