package ru.yandex.personal.mail.search.metrics.scraper.controllers;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.annotation.Retryable;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.controllers.mapping.SearchMapping;
import ru.yandex.personal.mail.search.metrics.scraper.metrics.MetricsCompatibleSystemManager;
import ru.yandex.personal.mail.search.metrics.scraper.metrics.basket.BasketQuery;
import ru.yandex.personal.mail.search.metrics.scraper.metrics.serp.Serp;
import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountConfiguration;
import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountManager;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerException;
import ru.yandex.personal.mail.search.metrics.scraper.services.scraping.crawling.CrawlerInternalException;

@Service("searchController")
public class SearchController implements SearchMapping {
    private final MetricsCompatibleSystemManager serviceManager;
    private final AccountManager accountManager;
    private final RetryTemplate retryTemplate;

    @Autowired
    public SearchController(
            MetricsCompatibleSystemManager serviceManager,
            AccountManager accountManager,
            RetryTemplate retryTemplate
    )
    {
        this.serviceManager = serviceManager;
        this.accountManager = accountManager;
        this.retryTemplate = retryTemplate;
    }

    @Override
    public Serp search(String system, String accountName, BasketQuery query) {
        return searchWithRetries(system, accountName, query);
    }

    @Override
    public Serp searchGet(String system, String accountName, String query) {
        return searchWithRetries(system, accountName, new BasketQuery(query));
    }

    private Serp searchWithRetries(String system, String accountName, BasketQuery query) {
        return retryTemplate.execute(context -> performSearch(system, accountName, query));
    }

    @Retryable(value = {CrawlerException.class, CrawlerInternalException.class})
    private Serp performSearch(String system, String accountName, BasketQuery query) {
        AccountConfiguration accountConfiguration = accountManager.getConfiguration(system, accountName);
        return serviceManager.getSystem(accountConfiguration).search(query);
    }
}
