package ru.yandex.personal.mail.search.metrics.scraper.metrics;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalListeners;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.services.account.AccountConfiguration;

@Service
public class CachedMetricsCompatibleSystemManager implements MetricsCompatibleSystemManager {
    private static final Logger LOG = LoggerFactory.getLogger(CachedMetricsCompatibleSystemManager.class);

    // reasonably big value, no need in configuration
    private static final int CACHE_MAX_SIZE = 100;

    private final MetricsSystemLoaderRegistry accountLoaderRegistry;
    private final LoadingCache<AccountConfiguration, MetricsCompatibleMailSystem> systemsCache;

    @Autowired
    public CachedMetricsCompatibleSystemManager(
            MetricsSystemLoaderRegistry accountLoaderRegistry,
            @Value("${scraper.accounts.connectionTtlSeconds}") Integer serviceTimeoutMinutes)
    {
        this.accountLoaderRegistry = accountLoaderRegistry;

        ExecutorService logoutExecutor = Executors.newCachedThreadPool();
        RemovalListener<AccountConfiguration, MetricsCompatibleMailSystem> removalListener =
                RemovalListeners.asynchronous(
                        notification -> {
                            LOG.debug("account timed out: " + notification.toString());
                            notification.getValue().logout();
                        }, logoutExecutor);

        systemsCache = CacheBuilder.newBuilder()
                .maximumSize(CACHE_MAX_SIZE)
                .expireAfterAccess(serviceTimeoutMinutes, TimeUnit.SECONDS)
                .removalListener(removalListener)
                .build(new CacheServiceLoader());
    }

    @Override
    public MetricsCompatibleMailSystem getSystem(AccountConfiguration loadingData) {
        LOG.debug("Service for account was requested " + loadingData);

        return systemsCache.getUnchecked(loadingData);
    }

    private class CacheServiceLoader extends CacheLoader<AccountConfiguration, MetricsCompatibleMailSystem> {
        @Override
        public MetricsCompatibleMailSystem load(AccountConfiguration config) {
            LOG.debug("Loading new service from registry " + config);
            MetricsSystemLoader loader = accountLoaderRegistry.getForInfo(config);
            return loader.load(config.getAccountPath());
        }
    }
}
