package ru.yandex.personal.mail.search.metrics.scraper.services.account;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mail.search.metrics.scraper.metrics.MetricsSystemLoaderRegistry;

@Service
public class AccountManagerImpl implements AccountManager {
    private static final Logger LOG = LoggerFactory.getLogger(AccountManagerImpl.class);

    private final Map<AccountInfo, AccountDraft> accountsToCreate = new ConcurrentHashMap<>();
    private final AccountRepository accountRepository;
    private final MetricsSystemLoaderRegistry accountLoaderRegistry;

    @Autowired
    public AccountManagerImpl(
            MetricsSystemLoaderRegistry accountLoaderRegistry,
            AccountRepository accountRepository)
    {
        this.accountRepository = accountRepository;
        this.accountLoaderRegistry = accountLoaderRegistry;
    }

    @Override
    public void initiateCreation(String system, String accountName, AccountProperties properties) {
        AccountInfo info = new AccountInfo(system, accountName);
        LOG.debug("Creation was initialized for " + info);

        if (accountsToCreate.containsKey(info) || hasAccount(info)) {
            LOG.info(info + " is already in use");
            throw new AccountException(info, "is already in use");
        }
        accountsToCreate.put(info, new AccountDraft(system, accountName, properties));
        LOG.debug(info.toString() + " was successfully created");
    }

    @Override
    public void addCredential(
            String system,
            String accountName,
            String credentialName,
            byte[] credentialBinary)
    {
        AccountInfo info = new AccountInfo(system, accountName);
        LOG.debug("Adding credential to " + info);

        validateCreationStage(info);
        validateCredentialName(info, credentialName);

        accountsToCreate.get(info).addCredential(credentialName, credentialBinary);
        LOG.debug("Part successfully added to " + info);
    }

    @Override
    public void finishCreation(String system, String accountName) {
        AccountInfo info = new AccountInfo(system, accountName);
        LOG.debug("Finishing account initialization " + info);

        validateCreationStage(info);

        accountRepository.writeAccount(accountsToCreate.get(info));
        accountsToCreate.remove(info);
        LOG.debug("Finished account initialization " + info);
    }

    @Override
    public void delete(String system, String accountName) {
        AccountInfo info = new AccountInfo(system, accountName);
        LOG.debug("Removing account " + info);
        validateHasAccount(info);
        accountRepository.deleteAccount(system, accountName);
        LOG.debug("Account was removed " + info);
    }

    @Override
    public AccountConfiguration getConfiguration(String system, String name) {
        AccountInfo info = new AccountInfo(system, name);
        LOG.debug("Config for account was requested " + info);
        validateHasAccount(info);
        return AccountConfiguration.fromInfo(info, accountRepository.getAccountPath(system, name));
    }

    @Override
    public boolean has(String system, String accountName) {
        return accountRepository.hasAccount(system, accountName);
    }

    private void validateHasAccount(AccountInfo info) {
        if (!hasAccount(info)) {
            throw new AccountException(info, "does not exists");
        }
    }

    private void validateCreationStage(AccountInfo info) {
        if (!accountsToCreate.containsKey(info)) {
            LOG.info("Account " + info + " is not in the creation stage");
            throw new AccountException(info, "is not in the creation stage");
        }
    }

    private void validateCredentialName(AccountInfo info, String credentialName) {
        if (!accountLoaderRegistry.getLoader(info.getSystemName()).validateCredentialName(credentialName)) {
            LOG.info("Attempt to add forbidden credential " + credentialName + " to " + info);
            throw new AccountException(info, "has no credential \"" + credentialName + "\"");
        }
    }

    private boolean hasAccount(AccountInfo info) {
        return has(info.getSystemName(), info.getAccountName());
    }
}
