package ru.yandex.personal.mail.search.metrics.scraper.services.account;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.NoSuchElementException;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Repository;
import org.springframework.util.FileSystemUtils;

@Repository
public class FileStorageAccountRepository implements AccountRepository {
    private static final Logger LOG = LoggerFactory.getLogger(FileStorageAccountRepository.class);
    private final Path baseDir;
    private final ObjectMapper mapper;

    @Autowired
    public FileStorageAccountRepository(
            @Value("${scraper.accounts.storagePath}") String baseDir,
            @Value("${scraper.persistentStoragePath:#{null}}") String persistentDir,
            ObjectMapper mapper
    )
    {
        if (persistentDir != null) {
            baseDir = persistentDir + "/" + baseDir;
        }
        this.baseDir = Paths.get(baseDir);
        this.mapper = mapper;
    }

    public FileStorageAccountRepository(String baseDir, ObjectMapper mapper) {
        this(baseDir, null, mapper);
    }

    @Override
    public boolean hasAccount(String system, String accountName) {
        return Files.exists(baseDir.resolve(system).resolve(accountName));
    }

    @Override
    public void deleteAccount(String system, String accountName) {
        if (!hasAccount(system, accountName)) {
            LOG.warn("Attempt to delete non existing account " + system + " " + accountName);
            throw new NoSuchElementException("No account " + system + " " + accountName);
        }

        boolean result = FileSystemUtils.deleteRecursively(baseDir.resolve(system).resolve(accountName).toFile());
        LOG.trace("Deleting account " + system + " " + accountName + " result: " + result);
    }

    @Override
    public void writeAccount(AccountDraft draft) {
        Path accDir = baseDir.resolve(draft.getSystemName()).resolve(draft.getAccountName());
        if (!Files.exists(accDir)) {
            createDirs(accDir);
        }

        writeProperties(accDir.resolve(AccountProperties.FILENAME), draft.getProperties());

        for (Map.Entry<String, byte[]> credentialEntry : draft.getCredentials().entrySet()) {
            LOG.trace("Writing account credential: " + credentialEntry.getKey());
            writeCredential(accDir.resolve(credentialEntry.getKey()), credentialEntry.getValue());
        }
    }

    @Override
    public Path getAccountPath(String system, String accountName) {
        return baseDir.resolve(system).resolve(accountName);
    }

    private void writeProperties(Path propertiesFile, AccountProperties properties) {
        try {
            mapper.writeValue(propertiesFile.toFile(), properties);
        } catch (IOException e) {
            LOG.error("Error writing account credential to file. Path " + propertiesFile, e);
            throw new UncheckedIOException(e);
        }
    }

    private void writeCredential(Path path, byte[] credentialBinary) {
        try {
            Files.write(path, credentialBinary);
        } catch (IOException e) {
            LOG.error("Error writing account credential to file. Path " + path, e);
            throw new UncheckedIOException(e);
        }
    }

    private void createDirs(Path dir) {
        try {
            Files.createDirectories(dir);
        } catch (IOException e) {
            LOG.error("Error creating account directory. Path " + dir, e);
            throw new UncheckedIOException(e);
        }
    }
}
