package ru.yandex.personal.mail.search.metrics.scraper.services.archive.storage;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.FileSystemUtils;

public class FileSystemTtlByteStorage implements ByteStorage {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemTtlByteStorage.class);

    private static final int FILE_NAME_LENGTH_THRESHOLD = 98;
    private static final String FILE_NAME_ELLIPSIS = "__";
    private static final DateTimeFormatter formatter = DateTimeFormatter.ISO_DATE;
    private static final String FORBIDDEN_PATH_CHARACTERS_REGEX = "[\\\\/:*?\"<>|]";
    private static final Duration INVALIDATION_DELAY = Duration.ofHours(4);

    private final Path baseDir;
    private final Integer storeDays;

    private Instant lastInvalidationAttempt = Instant.MIN;

    public FileSystemTtlByteStorage(Path baseDir, Integer storeDays) {
        this.baseDir = baseDir;
        this.storeDays = storeDays;
    }

    @Override
    public String save(byte[] content, String name) {
        checkInvalidation();
        Path dirToSave = currentDateDir();
        Path file;

        String fileNameWOExtension = shortenFileName(com.google.common.io.Files.getNameWithoutExtension(name));
        String cleanFileName = fileNameWOExtension.replaceAll(FORBIDDEN_PATH_CHARACTERS_REGEX, "_");
        String extension = com.google.common.io.Files.getFileExtension(name);
        try {
            file = Files.createTempFile(dirToSave, cleanFileName, "." + extension);
        } catch (IOException e) {
            LOG.warn("Can not create archive file for " + name + " in " + dirToSave.toString(), e);
            throw new UncheckedIOException(e);
        }

        try {
            Files.write(file, content);
        } catch (IOException e) {
            LOG.warn("Can not write archive file " + file.toString(), e);
            throw new UncheckedIOException(e);
        }

        return baseDir.toUri().relativize(file.toUri()).toASCIIString();
    }

    @Override
    public List<String> entriesNames() {
        if (Files.exists(baseDir)) {
            try {
                return Files
                        .walk(baseDir)
                        .filter(Files::isRegularFile)
                        .map(baseDir::relativize)
                        .map(Path::toString)
                        .collect(Collectors.toList());
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return Collections.emptyList();
    }

    @Override
    public byte[] getContent(String id) {
        try {
            return Files.readAllBytes(baseDir.resolve(id));
        } catch (IOException e) {
            LOG.warn("Can not read archive file: " + id, e);
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public boolean hasContent(String id) {
        return Files.exists(baseDir.resolve(id));
    }

    private String shortenFileName(String fileName) {
        if (fileName.length() > FILE_NAME_LENGTH_THRESHOLD) {
            LOG.info("File name is too long, reducing length to " + FILE_NAME_LENGTH_THRESHOLD);
            return fileName.substring(0, FILE_NAME_LENGTH_THRESHOLD) + FILE_NAME_ELLIPSIS;
        } else {
            return fileName;
        }
    }

    private void checkInvalidation() {
        Duration sinceLastInvalidationAttempt = Duration.between(lastInvalidationAttempt, Instant.now());
        LOG.debug("Time since last invalidation attempt: " + sinceLastInvalidationAttempt.toString());
        if (sinceLastInvalidationAttempt.compareTo(INVALIDATION_DELAY) < 0) {
            LOG.debug("Invalidation attempt was recently made, ignoring");
            return;
        }

        LOG.debug("Checking for invalid dates");
        if (Files.exists(baseDir)) {
            try {
                Files.walk(baseDir, 1).filter(dir -> !dir.equals(baseDir)).filter(Files::isDirectory).filter(dir -> {
                    String fileName = dir.getFileName().toString();
                    LocalDate date = LocalDate.parse(fileName, formatter);
                    boolean outdated = date.compareTo(LocalDate.now().minusDays(storeDays)) < 0;
                    if (outdated) {
                        LOG.debug("Directory " + dir.toString() + " is outdated with date " + date
                                + " and marked for deletion");
                    }
                    return outdated;
                }).forEach(this::deleteRecursivelyUnchecked);
            } catch (IOException e) {
                LOG.warn("Can not invalidate outdated archive directories", e);
            }
        }

        lastInvalidationAttempt = Instant.now();
    }

    private void deleteRecursivelyUnchecked(Path dir) {
        try {
            FileSystemUtils.deleteRecursively(dir.toFile());
            LOG.debug("Deleting folder recursively " + dir.toString());
        } catch (Exception e) {
            LOG.warn("Can not delete directory recursively " + dir.toString(), e);
        }
    }

    private void createDirs(Path dir) {
        try {
            Files.createDirectories(dir);
        } catch (IOException e) {
            LOG.error("Error creating account directory. Path " + dir, e);
            throw new UncheckedIOException(e);
        }
    }

    private Path currentDateDir() {
        LocalDate date = LocalDate.now();
        return dateDir(date);
    }

    private Path dateDir(LocalDate date) {
        String dateStr = formatter.format(date);
        Path dateDir = baseDir.resolve(dateStr);
        createDirs(dateDir);
        return dateDir;
    }
}
